/* Copyright (c) 2004, 2012, Oracle and/or its affiliates. 
All rights reserved. */
 
/* 
  NAME 
  ----
    clscrsx.h - Programming Interface (C API) to Cluster Ready Services 

  OVERVIEW
  --------
    CLSCRS is a set of Programming Interfaces (C APIs) to Oracle Clusterware.
    These APIs provide access to operational control of entities managed by the 
    clusterware, the entities being Resources, Resource Types, Servers, 
    Server Pools, and Server Categories. 
    The API is used to register user applications with Oracle
    Clusterware so that they can be managed by it and made highly available.
    Once an application is registered, it can be managed and its state queried.
    If the application is no longer to be run, it can be stopped and 
    unregistered from the clusterware. 

    Clusterware services are provided by Cluster Ready Services (CRS), that 
    is run as part of Oracle Clusterware

    This API employs a context explicitly manifested on all function calls. 
    The API does  not store or "anchor" anything at the process or thread level.
    Callbacks supplied in the API calls can be used for diagnostic logging.

  WHAT'S NEW?
  ----------
    In 12gR1, there are new CRS entity kinds and thus new APIs to manage them;
    additionally,  some of the pre-existing APIs have been deprecated. The new
    entities are Server Categories. Also, a new area of APIs was added: 
    CRS What-If (see "NEW IN 12.1" below).

    In 11gR2, there are new CRS entity kinds and thus new APIs to manage them;
    additionally, some of the pre-existing APIs have been revised. The new 
    entities are Resource Instances, Resource Types, Servers and Server Pools.
    This section summarizes the new, preserved, and deprecated APIs for 11gR1

    New 12gR1 APIs:
    -------------
     clscrs_request_action         -  action entrypoint for resource(s)
     clscrs_register_servercategory-  register server category(s)
     clscrs_unregister_servercategory-unregister server category(s) 
     clscrs_stat3                  -  query multiple entity kinds atomically 
     clscrs_stop_resource_in_pools -  stop resources in server pool(s)
     clscrs_start_resource_in_pools-  start resources in server pool(s)
     clscrs_restart_resource       -  stop+start running resource(s)
     clscrs_whatif_register_resource- evaluates the impact of registering a resource
     clscrs_whatif_fail_resource   -  evaluates the impact of a resource failure
     clscrs_whatif_start_resource  -  evaluates the impact of starting a resource
     clscrs_whatif_stop_resource   -  evaluates the impact of stopping a resource
     clscrs_whatif_relocate_resource    - evaluates the impact of relocating a 
                                          resource
     clscrs_whatif_register_serverpool  - evaluates the impact of registering a 
                                          server pool
     clscrs_whatif_unregister_serverpool- evaluates the impact of unregistering 
                                          a server pool
     clscrs_whatif_add_server      - evaluates the impact of adding a new server 
                                     to the cluster
     clscrs_whatif_relocate_server - evaluates the impact of relocating a server
     clscrs_whatif_delete_server   - evaluates the impact of a server failure
     clscrs_action_gettype         - return the type for the action
     clscrs_action_getentity       - return the entity for the action
     clscrs_action_getparams       - return the list of params for the action
     clscrs_actionlist_create      - create an actionlist
     clscrs_actionlist_first       - return the first action in an actionlist
     clscrs_actionlist_next        - return the next action in an actionlist
     clscrs_actionlist_count       - count the number of actions in the list
     clscrs_actionlist_seqid       - return the sequence ID for the list
     clscrs_actionlist_print       - print the actionlist
     clscrs_actionlist_destroy     - destroy the actionlist
     clscrs_crsentity_*            - a set of APIs to manage crs entities
     clscrs_crsentitylist_*        - a set of APIs to manage crs entity lists
     

    New 11gR2 APIs:
    -------------
     clscrs_register_type          -  register resource type(s)
     clscrs_register_serverpool    -  register a server pool
     clscrs_register_resource2     -  register resource(s)
     clscrs_start_resource2        -  start resources
     clscrs_stat2                  -  query resources
     clscrs_stop_resource2         -  stop resources
     clscrs_check_resource2        -  check resource(s)
     clscrs_relocate_resource2     - relocate resource(s)
     clscrs_unregister_resource2   -  unregister resource(s)
     clscrs_unregister_type        -  unregister resource type(s)
     clscrs_unregister_serverpool  -  unregister a server pool
     clscrs_type_create            -  create a resource type (allocates memory)
     clscrs_type_set_attr          -  set an attribute for a resource type
     clscrs_type_get_attr          -  get the value/properties of a resource 
                                      type attribute
     clscrs_compfilter_create      -  create a comparison filter
     clscrs_exprfilter_create      -  create an expression filter
     clscrs_filter_destroy         -  destroy a filter
     clscrs_relocate_server        -  relocate a server to a different
                                      serverpool
     clscrs_is_crs_admin           -  indicates if a user is a CRS Admin
     clscrs_get_entity_type        -  returns the clscrs_entity for the id 
                                      provided
     clscrs_get_resource_instance_details - gets the details of the resource
                                            instance id provided

    Pre-11gR2 APIs:
    ------------------

     clscrs_init_crs               -  init clscrs context
     clscrs_term_crs               -  term clscrs context
     clscrs_getnodename            -  get nodename for the current node
     clscrs_sp_set                 -  set the value for a stringpair (sp)
     clscrs_sp_get                 -  get the name and value for a stringpair
     clscrs_sp_get_value           -  get the value for a stringpair 
     clscrs_splist_create          -  create a list of stringpairs (splist)
     clscrs_splist_create_and_set  -  creat a splist and add a sp
     clscrs_splist_append          -  append a sp
     clscrs_splist_first           -  get the first sp in the splist
     clscrs_splist_next            -  get the next sp in the splist
     clscrs_splist_replace         -  replace the value for a sp
     clscrs_splist_delete_sp       -  delete the sp with given name (frees mem)
     clscrs_splist_find            -  find the sp matching the given name
     clscrs_splist_count           -  count the number of sps in a splist
     clscrs_splist_destroy         -  destroy a splist (frees memory)
     clscrs_crsentity_create       -  create an entity (allocates memory)
     clscrs_entity_id_create       -  create id for a clscrs entity
     clscrs_crsentity_get_name     -  get the name of an entity
     clscrs_crsentity_set_attr     -  set an attribute for an entity
     clscrs_crsentity_get_attr     -  get the value of an enity
     clscrs_crsentity_get_attr_list      -  get the attribute list for an entity 
     clscrs_crsentity_set_attr_list      -  set the attribute list for an entity
     clscrs_crsentity_set_crsentitylist  -  set the entity list for an entity
     clscrs_crsentity_get_crsentitylist  -  get the entity list for an entity
     clscrs_crsentity_attr_count   -  get the number of attributes for an entity
     clscrs_crsentity_get_op_status-  get the status of an operation on an entity 
     clscrs_res_get_node_list      -  get the hosting members for a resource
     clscrs_crsentity_destroy      -  destroy an entity
     clscrs_entity_id_destroy      -  destroy an entity id
     clscrs_crsentitylist_create   -  create a list of entities (entitylist)
     clscrs_crsentitylist_append   -  append an entity to a list
     clscrs_crsentitylist_first    -  get the first entity on the list
     clscrs_crsentitylist_next     -  get the next entity on the list
     clscrs_crsentitylist_find     -  find the entity matching the given name
     clscrs_crsentitylist_count    -  count the number of entities in the list
     clscrs_crsentitylist_delete_crsentity -  delete the entity matching the given 
                                              name
     clscrs_crsentitylist_destroy  -  destroy an entity list
     clscrs_get_fixed_attrlist     -  get the list of attributes corresponding 
                                      to an attribute group identifier
     clscrs_crsentity_get_registered     -  find out if an entity is registered

    These APIs are covered in detail further down in the document.

    These APIs can also be segregated based on whether they interact with CRS 
    or not.

    Interactive APIs:
    --------------------
 
    These APIs make a call to the CRSD asking it to execute commands on 
    their behalf. They require that the CRSD be online and running.

     clscrs_register_type          -  register resource type(s)
     clscrs_register_serverpool    -  register a server pool
     clscrs_register_resource2     -  register resource(s)
     clscrs_start_resource2        -  start resources
     clscrs_stat2                  -  query resources, resource types, 
                                      server pools, servers, serve categories
     clscrs_stop_resource2         -  stop resources
     clscrs_request_action         -  action entrypoint for resource(s)
     clscrs_check_resource2        -  check resource(s)
     clscrs_relocate_resource2     -  relocate resource(s)
     clscrs_unregister_resource2   -  unregister resource(s)
     clscrs_unregister_type        -  unregister resource type(s)
     clscrs_unregister_serverpool  -  unregister a server pool
     clscrs_relocate_server        -  relocate a server to a different 
                                      server pool
     clscrs_register_servercategory-  register server category(s)
     clscrs_unregister_servercategory-unregister server category(s) 
     clscrs_get_server_by_category -  get a list of servers that match a particular 
                                      server category.    
     clscrs_stat3                  -  query multiple entity kinds atomically 
     clscrs_stop_resource_in_pools -  stop resources in server pool(s)
     clscrs_start_resource_in_pools-  start resources in server pool(s)
     clscrs_restart_resource       -  stop+start running resource(s)
     clscrs_whatif_register_resource- evaluates the impact of registering a resource
     clscrs_whatif_fail_resource   -  evaluates the impact of a resource failure
     clscrs_whatif_start_resource  -  evaluates the impact of starting a resource
     clscrs_whatif_stop_resource   -  evaluates the impact of stopping a resource
     clscrs_whatif_relocate_resource    - evaluates the impact of relocating a 
                                          resource
     clscrs_whatif_register_serverpool  - evaluates the impact of registering a 
                                          server pool
     clscrs_whatif_unregister_serverpool- evaluates the impact of unregistering 
                                          a server pool
     clscrs_whatif_add_server      - evaluates the impact of adding a new server 
                                     to the cluster
     clscrs_whatif_relocate_server - evaluates the impact of relocating a server
     clscrs_whatif_delete_server   - evaluates the impact of a server failure
     

    Non-interactive APIs:
    -------------------

    These APIs are used for functions like: context initialization, 
    preparation of request payloads for interactive APIs and post-processing
    output of the interactive APIs. In short, they do not make any call to CRS.

    There is a new error reporting mechanism by way of callbacks that is 
    available for the new APIs to use. All interactive APIs, except, 
    clscrs_stat2 make use of this callback mechanism. It's used by clients of
    these APIs to receive error, warning and progress messages sent by the CRSD.
    Details about this mechanism are provided below in the "ERROR HANDLING"
    and "CALLBACK" section.

    Another new feature in this release is the use of filters to narrow down 
    the list of CRS entities to work upon. Details on this are given below in
    the "Filters" section.

  12.1 DEPRECATED STUFF
  ----------------

  a. clscrs_res & clscrs_reslist

    Previously, resource types, servers and server pools were referred to as
    clsrcs_res objects as part of the CLSCRS API. This does not make sense 
    anymore with addition of more CRS entity types, even as of 11gR2. In 12g, 
    all clscrs_res structures and their corresponding functions have been 
    deprecated. Instead, a new API object has been introduced named 
    clscrs_crsentity that is designed to represent all entity kinds managed by 
    CRS, irrespective of their types. All the functions that operated on 
    clscrs_res objects have been deprecated and replaced with new functions 
    that operate on the clscrs_crsentity object.

    Also, the clscrs_reslist object has been deprecated and replaced with the
    clscrs_crsentitylist to maintain consistency and readability with respect 
    to the addition of new CRS entities. All functions that operated on 
    clscrs_reslist objects have been deprecated and replaced with new functions 
    that operate on the clscrs_crsentitylist object.

  b. clscrs_entity

    Previously, clscrs_entity was the enumeration type used to identify the type
    of CRS entity. This has been deprecated and replaced by clscrs_entity_type. 


  11.2 DEPRECATED STUFF
  ---------------

    In 11g Release 2, some of the older APIs have been deprecated. These
    deprecated APIs, though supported in 11g Release 2, will not be supported
    in later releases. Below is a list of the deprecated APIs:

     clscrs_register_resource    -  register resource(s) DEPRECATED
     clscrs_start_resource       -  start resources DEPRECATED
     clscrs_stat                 -  query resources DEPRECATED
     clscrs_stop_resource        -  stop resources DEPRECATED
     clscrs_check_resource       -  check resource(s) DEPRECATED
     clscrs_relocate_resource    -  relocate resource(s) DEPRECATED
     clscrs_unregister_resource  -  unregister resource(s) DEPRECATED
     clscrs_msgf                 -  Callback function called to deliver 
                                    output text from CRS operations DEPRECATED
     clscrs_fail_resource        -  DEPRECATED

    The corresponding new APIs are mentioned below:

     clscrs_register_resource2   -  register resource(s), resource types
                                    and server pools
     clscrs_start_resource2      -  start resources
     clscrs_stat2                -  query resources, resource types, server
                                    pools and servers
     clscrs_stop_resource2       -  stop resources
     clscrs_check_resource2      -  check resource(s)
     clscrs_relocate_resource2   -  relocate resource(s)
     clscrs_unregister_resource2 -  unregister resource(s), resource types
                                    and server pools 
     clscrs_msgf 2               -  Callback function called to deliver output
                                    text from CRS operations.

  CONTEXT INITIALIZATION AND PERSISTENCE:
  --------------------------------------

    In order to use these APIs, the clscrs context must first be initialized.
    The calls to create/terminate this context are defined below:

     clscrs_init_crs             -  initialize clscrs context
     clscrs_term_crs             -  terminate  clscrs context
   
    The caller is responsible for terminating the context when its no longer 
    needed.

  THREADING SUPPORT:
  -----------------

    If initialized with the CLSCRS_FLAG_USETHREADS flag, the API may spawn 
    threads internally. Every API function executes in the context
    of the calling thread. The API context object may not be used concurrently
    by multiple threads; however, no thread-affinity on the part of the client 
    is required. A process may create multiple API contexts and use those on 
    different threads, subject to the one-thread-per-one-context-at-a-time rule.
    Note that if the application intends to interface with APIs on multiple
    threads, it must use the CLSCRS_FLAG_USETHREADS flag when initializing
    contexts.

  API DATA STRUCTURES:
  -------------------

    These are entities that will be passed into the API calls and will contain 
    return values from the API call.

   clscrs_sp:
     This is a string-pair (sp) structure. It contains a name and a value string
     The value may be NULL. It is created and destroyed, and its contents can
     be examined and the value replaced. A 'sp' may be a member of exactly one
     stringpair list (clscrs_splist).

   clscrs_splist:
     A stringpair list (splist) is a list of zero or more stringpairs used in
     various contexts. A stringpair list can be added to/removed from/ iterated
     over

   clscrs_crsentity:
     This represents a CRS entity, which contains the name and additional data 
     appropriate to the context in which it is used. Sometimes it contains CRS 
     entity attribute data and other times it carries status and return message 
     about an operation. An instance of clscrs_crsentity may be in exactly one 
     clscrs_crsentitylist.

   clscrs_crsentitylist:
     An entity list is a data structure used to contain zero or more instances 
     of clscrs_crsentity. 
     Like clscrs_splist, this also has addition/removal/iteration capabilities.

   Thread-Safety:
   -------------

     The routines provided to manage API data structures cannot be used with
     the same API context in multiple threads concurrently; however, no 
     thread-affinity on the part of the client is required. A process may
     invoke these routines on multiple threads provided that a separate
     API context is used in each instance.
     NOTE: The clscrs_init_crs and clscrs_term_crs APIs are not 
     thread-safe. If these APIs are used in multiple threads,
     calls to these 2 routines must be mutually exclusive.
     
  CONNECTION PERSISTENCE
  ----------------------
  
  As of 12.1, clscrs preserves the connection as long as the context is alive.
  The connection is not torn down after each call. Clscrs keeps the
  connection alive as long as the the context is valid.
  Clients should re-use the context instead of creating and destroying contexts
  for every call. 

  MEMORY MANAGEMENT:
  -----------------

    The API works on elements and lists. The elements are added to lists. The 
    memory for both is allocated and released through explicit API calls. It 
    is the caller's responsibility to release the memory that they allocate. 
    However, when elements are added to lists, only the list  must be destroyed: 
    the destruction of the list destroys its elements implicitly. The elements 
    must be destroyed  when they are not added to any list. For recursive lists, 
    destroying the parent list also destroys any lists contained within it. 
    clscrs_sp and clscrs_crsentity need to be destroyed by the caller. If they are 
    part of a clscrs_splist or clscrs_crsentitylist, destroying the list destroys the 
    respective clscrs_sp and clscrs_crsentity objects.

    For example, when an entity is created and added to a list, only the 
    list must be destroyed, but not the individual entity. Destroying the 
    list releases the memory for the individual entity(s) too.

    Memory is allocated by the API through the following calls -

     clscrs_sp_create()
     clscrs_crsentity_create()
     clscrs_serverpool_create()
     clscrs_type_create()
     clscrs_splist_create()
     clscrs_crsentitylist_create()
     clscrs_entity_id_create()
     
    Each of the above calls has a corresponding clscrs_xxx_destroy().

  ERROR HANDLING:
  --------------

    There are two error-handling mechanism employed by the APIs. Non-interactive 
    APIs use one and interactive, the other.

    For non-interactive APIs, the error code is returned as the return value of 
    the function call.
    For example, a non-interactive API like:

        clscrsret clscrs_sp_get_value(clscrs_sp *sp, oratext **value);

    The error code is returned as a type of "clscrsret" value.

    For interactive APIs, the output result is represented in the following 3 
    layers:

    1. The return value of the function call provides a high-level output of 
       the request (did the request reach the server? Was it completely 
       successful or completely /partially unsuccessful). A successful return 
       value means: the request was received, processed, and the outcome is 
       successful for all entities requested.

    2. For each entity the request operated on, there is a programmatic 
       completion code stored in the "op_status" list. If the value is not 
       success, it indicates the high level kind of the problem specific to 
       processing the request for the particular object.

    3. Optionally, the API may indicate it wants to receive localized 
       human-readable error/warning/status messages by using the callback 
       mechanism. Each invocation of the callback will provide the message, 
       message kind (severity) and the ID of the object to which the callback 
       invocation pertains.
      
    For example, an interactive API like:
      
       CLSCRS_STAT clscrs_register_resource2(clscrs_reslist *in_reslist, 
                                             uword flags,
                                             clscrs_msgf2 msgf, 
                                             void *msgarg,
                                             clscrs_reslist *op_status);

    1. Function returns error code as a type of "CLSCRS_STAT" value.

    2. CRSD sends error messages, warning messages and progress messages back 
       to the client through the "clscrs_msgf2" callback. The client needs to 
       implement the callback to process these messages returned by the CRSD.

    3. In earlier releases, the API also contained results of each operation on 
       the CRS entities as part of the "op_status" list. The way to access 
       that was through this API:

        clscrsret clscrs_res_get_op_status(clscrs_res *res, 
                                           CLSCRS_STAT *status,
                                           oratext **msg);

    The "status" argument contains a status code about the CRSD operation on 
    the CRS entity. Also, the "msg" argument contains a message from the CRSD 
    about the result of the operation. Though the "op_status" list continues 
    to contain the results of the CRSD operation for each CRS entity in the
    "msg" argument, usage of the "msg" argument to get the error codes/messages 
    has now been DEPRECATED and is not supported for any use of the API on a 
    new entity; only pre-existing use cases (for acting on resources 
    specifically) are supported. Please make use of the callback function to 
    process any messages returned by the CRSD. See "CALLBACK" section for 
    more information.

  EXCEPTION PROPAGATION
  ---------------------
    Every API uses thread-local storage for clsk exception propagation.
    This API returns the error stack that caused a particular API to fail.
    To get the error stack, users should use clscrs_get_error_details() API.
    Users would be able to figure out what caused the API to fail using the
    above API.
    Note that this is in addition to the methods mentioned in the ERROR HANDLING
    section, and is not a replacement for those methods.

  CALLBACK MECHANISM and TRACING:
  ------------------------------

    Interactive APIs in CLSCRS provide a callback mechanism that the clients 
    can use to process error messages, warning messages and progress messages 
    sent by the CRSD.

    The signature of this callback function is given below:

      typedef  void (*clscrs_msgf2)(void *usrp, 
                                    const oratext *id, 
                                    const oratext *msg,
                                    clscrs_msgtype msgtype);

    where:
      usrp    -- user supplied pointer, probably containing context of the call.
      id      -- The identifier of the entity to which the msg corresponds to
      msg     -- output text
      msgtype -- type of the message - progress, warning or error

    A sample callback function is shown below:

    void myCallback(void *arg, const oratext *pId, const oratext *pMsg,
                            clscrs_msgtype msgType)
    {
       if (pMsg != NULL)
       {
           cout << pMsg << endl;
       }
    }

    Below is an example on how to use this callback function in an interactive API:

      clscrs_start_resource2(pResIdList, NULL,
                             env, myCallback, NULL,
                             0, pOpStatus);

    There is also a way to output debug trace messages for the API itself. That 
    is achieved by passing the CLSCRS_FLAG_TRACE flag when creating the context. 
    The signature for context creation is:

      CLSCRS_STAT clscrs_init_crs(clscrs_ctx **ctx, 
                                  clscrs_msgf errf, 
                                  void *errCtx,
                                  ub4 flags);

    For the trace messages to work, both the trace flag and a callback function 
    of clscrs_msgf type must be specified in the clscrs_init_crs API.

    The clscrs_msgf callback has the following signature:

      typedef void (*clscrs_msgf)(void *usrp, const oratext *msg, sword msglen);


  FILTERS:
  -------

    A new feature in 11g Release 2 is the use of filters to narrow down CRS 
    entities that need to be operated upon. A filter is conceptually a 
    condition. Simple Filters are attribute-value pairs with a comparator. 
    Simple Filters can be combined into expressions (a.k.a Expression Filters)
    using Boolean operators.

    The filter comparators that are supported are: 

                =,  >,  <,  !=,  co,  st,  en, nco, coi, eqi 

    The comparators: =, >, <, and != are self-explanatory. 
    Other comparators have the following meaning:
 
         co   -  stands for "contains" like NAME co db
         st   -  stands for "starts with" like NAME st ora
         en   -  stands for "ends with" like NAME en type
         nco  -  stands for "does not contain" like NAME nco ora
         coi  -  stands for "contains case-insensitive" like NAME coi oRa 
         eqi  -  stands for "equals case-insensitive" like NAME eqi oRa 
                      
    The Boolean operators that are supported are: 

                AND and OR

    Sample filters:
    1. TYPE = type1
    2. ((TYPE  =  type1) AND (CHECK_INTERVAL  >  50))
    3. (TYPE  =  type1) AND ((CHECK_INTERVAL  >  30) OR (AUTO_START co never))
    4.  NAME en network.res
    5.  TYPE st ora.db

    Use the clscrs_comparator enum to get the correct type for the above 
    comparators in the API calls.

    Use the clscrs_operator enum to get the correct type for the above 
    operators in the API calls.

    There are 2 kinds of filters, and the CLSCRS API has a set of APIs to 
    create these filters:

    1. Comparison Filter:
    ----------------------

      This is a simple filter that compares two values. 
      For e.g., TYPE  =  ora.db.type

      The API to create this is:

        clscrs_compfilter_create

    2.  Expression Filter:
    -----------------------

    This is a filter that is created from a set of comparison filters and/or 
    expression filters. Essentially, it is a nested set of filters.

    For e.g., ((TYPE  =  ora.db.type) AND (CHECK_INTERVAL  >  50))

    The API to create this is:

      clscrs_exprfilter_create

    NOTE: Both these APIs allocate memory and must be freed by calling  
          clscrs_filter_destroy().

    Filters can be used in most of the interactive APIs in place of an entity 
    list.
    Below is a short example to create a comparison filter: (TYPE = ora.db.type)

      clscrs_compfilter_create(ctx, 
                               clscrs_TYPE,
                               clscrs_comparator_eq, 
                               (const oratext *)"ora.db.type",
                               &myCompFilter);


    Another example below shows how to create an expression filter:

      clscrs_exprfilter_create(myCompFilter1,
                               clscrs_operator_or,
                               myCompFilter2, 
                               &myExprFilter);

    Below is an example on how to use these filters in an interactive API:

      clscrs_start_resource2(myCompFilter, NULL,
                             env, msgf2, NULL,
                             0, pOpStatus);

  VERSIONING:
  -----------

  Unless an API header has a SINCE section, all the APIs are available since
  CRS 10.2, except for the ones mentioned below:

  1. All clscrs_query_* APIs are available since CRS 12.1
  2. All clscrs_crsentity_* APIs are available since CRS 12.1
  3. All clscrs_querylist_* APIs are available since CRS 12.1
  4. All clscrs_crsentitylist_* APIs are available since CRS 12.1
  5. All clscrs_action_* APIs are available since CRS 12.1
  6. All clscrs_actionlist_* APIs are available since CRS 12.1

  Also, see section "WHAT'S NEW".

  RESOURCES AND RESOURCE INSTANCES SINCE 11g RELEASE 2 ONWARDS:
  -------------------------------------------------

  Pre 11g Release 2, CRS only supported one kind of a resource - application - 
  which had a single instantiation within the cluster. Such resource kind is 
  often referred to as a "Singleton" In 11g Release 2 the built-in 
  cluster_resource and local_resource types introduce the notion of plurality 
  to all resources: a resource may have one or more instantiation per node in 
  the cluster. To support this new model, the notion of a Resource Instance 
  (a.k.a. Resource Instantiation) was introduced in 11g Release 2 of the API 
  whereby all state data is kept on resource instances, while the configuration 
  is generally created and managed per-resource. Resource instances can be 
  queried, started, stopped, and relocated. They are created at resource 
  registration time and removed at resource un-registration implicitly. Resource 
  Instances have a unique ID by which they can be referred to in API calls (the 
  ID can be queried using the clscrs_stat2 API or alternatively, filters can be 
  used to operate on instances implicitly).

  AGENT USAGE:
  -------------------------
    When using clscrs APIs inside the agent entry points, there
    are a few considerations to keep in mind:
   
    1. Some actions (start/stop/clean) are executed under a lock on the resource
       instance. Thus, issuing a request to the server to act on the resource
       directly or by extension of a relation will be a dead-lock.
   
    2. Issuing read-only commands (clscrs_stat2) is generally safe UNLESS it's
       an "initial check", where the agent must not call back on the server as 
       that will dead-lock as well. Check type can be queried through
       clsagfw APIs.
  

  EXAMPLES
  --------

    Refer to the demo application <CRS Home>/crs/demo/crsapp.c

  WHAT-IF FUNCTIONALITY
  ---------------------
  In 12g Release 1m the CLSCRS What-If API was added to to describe Clusterware's response 
  to a hypothetical planned or unplanned event. 

  Clusterware already has the ability to react to events 
  in the system, by producing a response action plan. This action plan consists of a 
  series of resource state transitions and/or server pool reconfigurations. The What-If 
  API provides a mechanism to expose this action plan externally, to allow clients to 
  predict the consequences of an event before it actually takes place. 

  What-If response plans are available for the following event categories:
    (1) Resources       (Start/Stop/Relocate/Add/Modify/Fail)
    (2) Server Pools    (Add/Remove/Modify)
    (3) Servers         (Add/Remove/Relocate)
    (4) Policy          (Change Active Policy)
    (5) Server Category (Modify)

  The What-If output is provided as a list of "actions", with each action representing a 
  specific step that would be taken by CRS. Each action is encapsulated by a clscrs_action
  struct, and the clscrs_actionlist struct represents the entire sequence of actions.
  A set of functions (clscrs_action_* for action structs and clscrs_actionlist_* for 
  actionlists) are also provided to create, examine, iterate over and destroy these
  structures. Their usage is identical to the corresponding entitylist and entity functions.

  The What-If API also provides clients with the ability to make a query on the projected
  state of the system. The query is specified using the new stat3 format, with a 
  clscrs_querylist structure, and the result is provided by a clscrs_crsentitylist structure.
  Refer to the stat3 section for details on their usage.

  Each what-if response provided by CRS includes a sequence ID, which indicates the current 
  state of CRS. This ID is incremented for every new event handled by CRS. CRS guarantees that,
  as long the sequence ID has not changed, the action plan provided will be executed as-is for
  the event in question. For example, the action plan provided for a "whatif start resource FOO" 
  request would be identical to the actions CRS would take when an actual "start resource FOO"
  request is submitted, provided the sequence ID has not changed in the meantime.

  Sample Usage of API:

      boolean           tracectx = TRUE;
      oratext           *resid;
      clscrs_ctx        *ctx;
      clscrs_env         env;
      clscrs_splist     *resid_list;
      clscrs_action     *cur_actn;
      clscrs_actionlist *alist;
      clscrs_splist     *params;

      // Init crs 
      clscrs_init_crs(&ctx, (clscrs_msgf)clsuslztrace, &tracectx, (ub4)0);

      // Init parameters to the call
      clscrs_entity_id_create(ctx, "MYRES", clscrs_entity_res, &resid);
      clscrs_splist_create(ctx, &resid_list);
      clscrs_splist_append(resid_list, resid, NULL);
      clscrs_actionlist_create(ctx, &alist); 

      // Make call into the what-if API
      clscrs_whatif_start_resource(resid_list, nodename, flags, NULL, NULL, alist);

      // Process the resulting list of actions 
      for(clscrs_actionlist_first(alist,&cur_actn);cur_actn;clscrs_actionlist_next(alist,&cur_actn)) 
      {
        params = clscrs_action_getparams(cur_actn);
        switch(clscrs_action_gettype(cur_actn))
        {
          case clscrs_actiontype_resstate:
               // Read params and do something 
                  break;
          case clscrs_actiontype_srvmove:
               // Read params and do something
                  break;
          case clscrs_actiontype_newgroup:
               // Read params and do something
                  break;
          case clscrs_actiontype_errorcase:
               // Read params and do something
                  break;
        }
      }
      clscrs_actionlist_destroy(alist);
      clscrs_splist_destroy(resid_list);
      clscrs_term_crs(&ctx);


  SERVER CATEGORIZATION 
  ---------------------    

  In 12gR1, CRS introduced the notion of server categorization (a.k.a server
  coloration) - a mechanism that allows servers to be categorized/colored and
  server pools to restrict which server color(s) they accept. 

  Specifically, the mechanism functions thus:

    * Every server has a collection of well-known attributes, which can be
      either auto-discovered or user-assigned. Memory size and CPU count are
      examples of the auto-discovered kind, while a numerical value reflecting a
      relative preference in picking servers to move between pools during automatic
      server pool reconfigurations is an example of a user-assigned one (a user in
      this context can be an automated subsystem external to CRS, such as WLM)

    * A new CRS-managed entity kind named - servercategory - is introduced, which
      represents a Boolean-like expression that uses attributes to specify
      conditions that allows/disallows servers to be treated as belonging to the
      category/color.

    * A new attribute  SERVER_CATEGORY - is added to the server pool entity kind. A
      category can then be assigned to a server pool, as a new attribute of the
      Server Pool entity, which would indicate that only servers that are passing
      the category's conditions are valid candidates to be in the pool. This 
      attribute is also present in resources.

  A new CRS entity, called clscrs_entity_server_category, has been created and is
  part of the clscrs_entity_type enum.

  The following CLSCRS APIs are provided to manage Server Categories:

    a. clscrs_servercategory_create
    b. clscrs_servercategory_destroy
    c. clscrs_register_servercategory
    d. clscrs_unregister_servercategory
    e. clscrs_get_server_by_category
    f. clscrs_register_server
  

  STAT3
  -----
  Starting with 11gR2, CRS started managing entities of several kinds (resources,
  server pools, etc). However, the existing interfaces allow retrieving (reading)
  entities by kind, meaning that a single retrieval can only return entities of a
  single kind. Therefore, clients that need to get entities of different kinds and
  have a consistent view of the data structures to make further decisions had had
  to rely on a workaround using a special event sequence id and, if necessary,
  re-issue query requests several times (in theory, in a system with ever changing
  state/configuration such a solution is time-unbounded). 

  In 12g, CRS added a mechanism to perform a consistent read of entities of
  several kinds. This mechanism works on entities transparently, such that
  addition of new managed entities does not require any changes to the very
  mechanism. 

  This is achieved by making use of the following CLSCRS API:

    clscrs_stat3

  EXISTING API CHANGES
  --------------------

  a. clscrs_stat2
  
    New flags have been added to this API:
    
    CLSCRS_STATFLAG_SERVERBYCATEGORY
    
      Used to to query the servers that match a particular
      server category.
    
    CLSCRS_STATFLAG_CATEGORYBYSERVER
    
      Used to query the server categories that match a particular server.
  
  b. clscrs_start_resource2 & clscrs_stop_resource2
    
    New flags have been added to these APIs:
      
    CLSCRS_FLAG_T_HA_PREPARE
    
      Used in Transparent High Availability (start2/stop2 APIs) to instruct 
      CRSD to prepare for an external management interface's acting on a 
      resource. When the call comes back, if successful, the interface then 
      is expected to start/stop the resource and call back with the other flag
      (CLSCRS_FLAG_T_HA_FINISH).
                            
    CLSCRS_FLAG_T_HA_FINISH
                            
      Used in Transparent High Availability in conjunction with 
      CLSCRS_FLAG_T_HA_PREPARE. 
      Must be used in the second call to crsd (start2/stop2 APIs) when the
      action (start or stop) has finished.
      Note that clients must indicate resource instances and never resources 
      with this flag, as CRSD must know which instance(s) the invocation of
      the API applies.

    CLSCRS_NO_STATE_RESTORATION

      Available for use for start/stop resource APIs to instruct crsd to skip
      resource state restoration of the resources whose state was affected.
      That procedure is usually attempted unless this flag was used.
                                                                            

  RELATED DOCUMENTS
  -----------------
    (1) Oracle Clusterware Administration and Deployment Guide - 12g Release 1
    (2) Oracle Technology Network - http://www.oracle.com/technology
       

  MODIFIED   (MM/DD/YY)
   agusev      09/26/12 - fixed a typo
   rtamezd     09/17/12 - Fix bug 14115195
   agusev      06/07/12 - 12.1 doc update
   agusev      05/22/12 - fix 14023791
   nbraak      04/30/12 - pool dispersion
   gdbhat      04/25/12 - Bug 14000182
   gdbhat      04/20/12 - Bug 13694990
   anjiwaji    04/16/12 - Add clscrs_VOLUME_DEVICE.
   mrmehta     04/12/12 - Bug 9338300
   mrmehta     03/26/12 - bug 13838972 and 13680982
   agusev      03/19/12 - fix for 13862308
   gdbhat      03/13/12 - Bug 13573192
   gdbhat      02/28/12 - Bug 13787624
   gdbhat      02/19/12 - Bug 13084355
   agusev      02/14/12 - added RESOURCE_USE_ENABLED SA
   gdbhat      02/08/12 - Add clscrs_string_to_filter
   agusev      01/03/12 - added DEPLOYMENT SA
   gdbhat      12/07/11 - Update thread safety section
   mrmehta     10/26/11 - Bug 13095535
   nbraak      10/07/11 - crs hub size
   agusev      10/04/11 - fixed lang constant
   csivanan    09/19/11 - auto mode integration
   gdbhat      09/16/11 - Add Generic Application type
   csivanan    09/12/11 - auto mode integration
   nbraak      09/01/11 - resource alert log monitoring
   gdbhat      08/25/11 - Add CRSADMIN statflag
   csivanan    07/06/11 - resource restart op
   vmanivel    06/17/11 - Blocking Stack Start
   nbraak      07/12/11 - suppress ohasd driver restart
   vmanivel    06/02/11 - WhatIf for Cluster Admins
   csivanan    05/11/11 - Server pool: res start/stop
   gdbhat      04/30/11 - Tag APIs with CRSD version required
   vmanivel    04/11/11 - WhatIf activate policy
   nbraak      04/06/11 - action entrypoint
   vmanivel    03/03/11 - WhatIf for add/modify/fail
   vmanivel    02/12/11 - Adding USER_WORKLOAD attribute
   vmanivel    02/01/11 - WhatIf for Servers
   vmanivel    02/03/11 - Adding WhatIf API calls
   agusev      02/02/11 - T-HA API
   gdbhat      01/20/11 - Added clscrs_crsentity_get_type
   gdbhat      01/03/11 - Server Category changes
   vmanivel    12/21/10 - WhatIf for ServerPools
   vmanivel    12/21/10 - WhatIf stat3
   gdbhat      11/25/10 - Policy mgmt APIs
   nbraak      11/01/10 - adding new attributes: clscrs_DELETE_TIMEOUT, 
                          clscrs_MODIFY_TIMEOUT for delete/modify entrypoints.
   agusev      06/08/10 - XbranchMerge stpolu_b-8931907 from st_has_11.2.0.1.0
   gdbhat      11/16/10 - Add policy mgmt stuff
   vmanivel    10/01/10 - What-If for Resources
   gdbhat      09/28/10 - Added code for stat3
   agusev      06/08/10 - XbranchMerge stpolu_b-8931907 from st_has_11.2.0.1.0
   vmanivel    12/01/09 - Bug 9107142
   agusev      11/13/09 - added internal state constants
   gdbhat      10/07/09 - Bug 8812969  
   stpolu      10/05/09 - Bug 5878410
   stpolu      09/25/09 - Bug 8931907
   stpolu      09/04/09 - Bug 8703187
   gdbhat      05/19/09 - Bug 8514695
   vmanivel    05/11/09 - Bug 8258489, removing language references
   lmortime    04/28/09 - Adding LOGGING_LEVEL
   anutripa    04/22/09 - Add clscrs_reslist_isempty
   gdbhat      04/15/09 - Updating header
   agusev      04/08/09 - Added AUTOSTART QUEUED as a state details value for
                          server
   lmortime    04/07/09 - Added CRS_ORIGINAL_REASON
   agusev      03/06/09 - fix to 7827811
   agusev      02/23/09 - Removed CLSCRS_ATTRVAL_MAXLEN
   agusev      01/12/09 - Added server state constants.
   agusev      12/09/08 - Added what-if flag
   lmortime    11/19/08 - Redoing long view name transaction (bug 7143372) with
                          fix for lrgem
   lmortime    11/13/08 - rolling back 7143372
   agusev      11/12/08 - Noted the message part of op_status has been
                          deprecated
   lmortime    09/30/08 - Bug 7143372 - Fixing long connstr problems
   agusev      09/23/08 - added missing DEPRECATED tags
   hkanchar    09/16/08 - 
   lmortime    09/08/08 - Bug 7293087 - Removing deprecated servergroup calls
   agusev      08/08/08 - fixed the description of the QUEUE flag
   gdbhat      08/06/08 - Bug 6838123
   hkanchar    08/04/08 - 
   agusev      07/21/08 - Added CRS_SCRIPT_SUFFIX special value
   sltam       07/21/08 - Add CLSCRS_STATFLAG_NONE
   lmortime    07/21/08 - Bug 7142054 - changing server groups to server pools
   lmortime    07/18/08 - Bug 6725846 - Renaming quandary to intermediate
   agusev      06/20/08 - Added CRS_NLS_LANG
   gdbhat      06/17/08 - Bug 6957064. Add clscrs_check_resource2
   gdbhat      05/29/08 - Bug 7111562
   agusev      05/27/08 - Fixed comments for the FORCE flag
   agusev      04/09/08 - Changed message classification enum to match the
                          severity values set by RAS-WG
   agusev      03/04/08 - added more comments for the purge flag
   agusev      02/22/08 - Changed values for new return codes
   gdbhat      02/21/08 - Bug 6616277 - Add CLSCRS_STATFLAG_EVAL
   gdbhat      02/07/08 - Bug 6805819
   gdbhat      02/05/08 - Bug 6722729
   agusev      02/01/08 - Added CRS_EXE_SUFFIX special value
   gdbhat      01/31/08 - Bug 6756966
   akhaladk    01/14/08 - review APIs. bug 6118023
   skakarla    12/14/07 - rolling upgrade changes
   rvenkate    12/14/07 - add clscrs_DEFAULT_TEMPLATE
   agusev      12/13/07 - added CRS_ONLINE_CSS_NODENAMES
   gdbhat      11/16/07 - Add clscrs_res_set_reslist
                          clscrs_res_get_reslist
   akhaladk    10/24/07 - rename name tag for type
   gdbhat      10/23/07 - Add is_crs_admin
   akhaladk    10/23/07 - Add separate acl tag for resource types.
   gdbhat      10/19/07 - Add MAXOCRLEN
   agusev      10/19/07 - Added a reserved value definitions 
   gdbhat      10/05/07 - Backout to ver. 47
   gdbhat      09/22/07 - Add comparators and operators
   akhaladk    09/11/07 - Add some more attributes
   agusev      09/11/07 - Adding flags for transparent HA and stop that keeps
                          targets
   gdbhat      09/04/07 - Add relocate_server
   agusev      07/30/07 - Adding STATE to Server
   gdbhat      08/27/07 - Add purge flag
   jgrout      08/23/07 - Add unique AGFW external attrs,
                          fix CRS attrs and const correctness
   agusev      07/15/07 - Renamed SERVER_GROUP to SERVER_GROUPS
   agusev      07/19/07 - Fixed a missing quote
   agusev      06/18/07 - Removed duplicate definitions
   rdasari     06/11/07 - add ALL flag
   rdasari     05/21/07 - add CLSCRS_STATFLAG_FULLTYPE
   rdasari     04/12/07 - add more filter operators
   rdasari     03/27/07 - query resource instances
   agusev      03/05/07 - Added state constants
   agusev      02/13/07 - Adding special values
   rdasari     02/22/07 - add Memory Management section
   agusev      01/31/07 - Fixing comments
   rdasari     01/22/07 - add CONFIG flag for type attrs
   rdasari     01/09/07 - add relocate api
   akhaladk    12/26/06 - Add new codes for ACL errors.
   agusev      12/19/06 - Adding new attributes
   rdasari     11/27/06 - add trace functions for getting progress messages
   agusev      11/16/06 - Adding ALIAS_NAME
   rdasari     10/09/06 - add type_get_attr
   agusev      09/21/06 - Adding new error codes
   vrai        08/31/06 - Changes for clsinit
   rdasari     09/18/06 - change register_server_group signature
   rdasari     09/12/06 - add flags to type_set_attr
   rdasari     06/01/06 - add filter api, clscrs_stat2
   akhaladk    04/14/06 - fix compile error 
   rdasari     03/28/06 - add new 11g api signatures 
   rdasari     05/06/05 - document CLSCRS_FLAG_USETHREADS
   rdasari     03/23/05 - add new attribute UPTIME_THRESHOLD
   agusev      10/29/04 - Adding comments for the new flag 
   agusev      10/27/04 - Adding the EVENT flag 
   rdasari     08/03/04 - rdasari_api_app
   rdasari     07/27/04 - Creation

*/

#ifdef __cplusplus
extern "C" {
#endif

#ifndef ORATYPES
# include <oratypes.h>
#endif

#ifndef CLSCRSX_ORACLE
# define CLSCRSX_ORACLE

/* Return codes for the CLSCRS api */

typedef enum
{
  CLSCRS_STAT_SUCCESS=0,                               /* "Command Success" */ 
  CLSCRS_STAT_CONTINUE,                                         /* try again */
  CLSCRS_STAT_CONNECTION=184,                /* Cannot communicate with crsd */
  CLSCRS_STAT_INVALID_ARGS=188,                  /* Invalid arguments passed */
  /* The input entity list does not match with the api */
  CLSCRS_STAT_ENTITY_MISMATCH=189, 
  CLSCRS_STAT_WARNING=199,        /* "Catch all warning. No reason provided" */
  CLSCRS_STAT_FAILURE=200,        /* "Catch all failure. No reason provided" */
  CLSCRS_STAT_BAD_COMMAND=201,    /* "Catch all failure. No reason provided" */
  CLSCRS_STAT_NO_APPS=202,                  /* "No resources are registered" */
  CLSCRS_STAT_OTHER=203,                                 /* "To be supplied" */
  CLSCRS_STAT_ATTR_NOTFOUND=204,                      /* Attribute not found */
  CLSCRS_STAT_APP_NAME=210,                     /* "Could not find resource" */
  CLSCRS_STAT_IS_REGISTERED=211,   /* "Resource has already been registered" */
  CLSCRS_STAT_NOT_REGISTERED=212,            /* "Resource is not registered" */
  CLSCRS_STAT_REGISTER_FAIL=213,            /* "Could not register resource" */
  CLSCRS_STAT_UNREGISTER_FAIL=214,        /* "Could not unregister resource" */
  CLSCRS_STAT_START_FAIL=215,                  /* "Could not start resource" */
  CLSCRS_STAT_STOP_FAIL=216,                    /* "Could not stop resource" */
  CLSCRS_STAT_RELOCATE_FAIL=217,            /* "Could not relocate resource" */
  CLSCRS_STAT_NOT_RESTART=218,               /* "Could not restart resource" */
  CLSCRS_STAT_UPDATE_FAIL=219,                /* cannot update the resource. */
  CLSCRS_STAT_PROFILE_INVALID=220,               /* Your profile was invalid */
  CLSCRS_STAT_PROFILE_BADPATH=221,      /* The profile was not found on disk */
  CLSCRS_STAT_DEPENDENCY=222,                         /* dependency problem. */
  CLSCRS_STAT_PLACEMENT=223,                           /* placement problem. */
  CLSCRS_STAT_SCRIPT_OUTPUT=224,           /* Warning: failed to redirect IO */
  CLSCRS_STAT_MISSING_RESNAME = 225,        /* Resource name not specified.  */
  CLSCRS_STAT_MEMALLOC_WARNING = 226, /* Internal memory allocation warning. */
  CLSCRS_STAT_INVALID_RESNAME = 227,             /* Invalid Resource name.   */
  CLSCRS_STAT_INVALID_HOST = 228,                      /* Invalid Hostname.  */
  CLSCRS_STAT_INVALID_FLAGS = 229,         /* "Invalid combination of flags" */
  CLSCRS_STAT_MEMBER_INVALID=230,          /* member is not a cluster member */
  CLSCRS_STAT_OPERATION_INVALID=231,/* "operation invalid for resource type" */
  CLSCRS_STAT_MEMBER_DOWN=232,                             /* member is down */
  CLSCRS_STAT_RESOURCE_BUSY=233,                         /* resource is busy */
  CLSCRS_STAT_USEROPTS_INVALID=234,  /* User supplied attribs not in profile */

  /* above ends the 9.2 set of messages */

  CLSCRS_STAT_INVALID=239,                                /* Invalid command */
  CLSCRS_STAT_CHANNEL_OUTPUT=240,              /* Piped channel output error */
  CLSCRS_STAT_INVALID_DIRPATH=241,                /* Invalid directory path. */
  CLSCRS_STAT_ACL_UNAVAILABLE = 242,     /* ACL information is not available */
  CLSCRS_STAT_DEFAULT_SETTING_UNAVAILABLE = 243,/*default setting unavailable */
  CLSCRS_STAT_REGISTER_UNAVAILABLE = 244,         /* registration allow/deny */
                                             /* information is not available */
  CLSCRS_STAT_NO_PRIVILEGE = 245,   /* User doesn't have enough privilege to */
                                                    /* perform the operation */
  CLSCRS_STAT_INVALID_ID = 246,                        /* User id is invalid */
  CLSCRS_STAT_INVALID_NAME = 247,                    /* User name is invalid */
  CLSCRS_STAT_ACL_FAILED = 248,                      /* Acl operation failed */
  CLSCRS_STAT_AUTHENTICATION_FAILURE=249,        /* authentication failure.  */
  CLSCRS_STAT_CONFIGURE_FAILURE=250,/* Configuration unsuccessful. Try again */
                                                        /* or restart caads. */
  CLSCRS_STAT_CONFIGURATION_MISS=251,     /* CAA configuration file missing. */
  CLSCRS_STAT_CONFIGURATION_ERROR=252,      /* CAA configuration file syntax */
                                                                   /* error. */
  CLSCRS_STAT_MISSING_USR_DEF_DIR=253,     /* USER_DEFAULT_DIR is not set in */
                                                 /* CAA configuration file.  */
  
  CLSCRS_STAT_AUTHORIZATION_FAILURE=254,            /* authorization failure */
  CLSCRS_STAT_USRNAME_CONFLICT=256,          /* username conflicts with owner*/
  CLSCRS_STAT_GRPNAME_CONFLICT=257,  /* groupname conflicts with primary group*/ 
  CLSCRS_STAT_UNKNOWN_ACL=258,                          /* unknown acl string */
  CLSCRS_STAT_NO_GRP_MEMBERSHIP=259, /* owner does not belong to the group    */

  /* Do not specify return codes exceeding 255 for CRS */ 

  /* messages 301 - 324 in use */

  /* messages 325 - 350  Client side resource validation */
  CLSCRS_STAT_RESNAME_MISMATCH=325,
  CLSCRS_STAT_INVALID_RESTYPE=326,
  CLSCRS_STAT_MISSING_ATTR=327,
  CLSCRS_STAT_ACTSCR_NOTEXIST=328,
  CLSCRS_STAT_HOSTMEM_REQ=329,
  CLSCRS_STAT_HOSTMEM_NOTREQ=330,
  CLSCRS_STAT_INVALID_HOSTMEM=331,
  CLSCRS_STAT_INVALID_PLACEMENT=332,
  CLSCRS_STAT_INVALID_AUTOSTART=333,
  CLSCRS_STAT_INVALID_VALUE=334,
  CLSCRS_STAT_RESLIST_NOTEMPTY=335,
  
  /* messages 401 - 420 in use */
  CLSCRS_STAT_CSSINIT_FAILED=413,                         /* CSS init failed */

  /* Messages 451 - 470 OCR related */
  CLSCRS_STAT_NO_OCR=451,                     /* "Can't find initialize OCR" */
  CLSCRS_STAT_NO_CRSD_CONN=452,             /* "Can't find crsd conn in OCR" */
  CLSCRS_STAT_INST_NOT_ENABLED = 453,             /* Instance is not enabled */

  /* Messages 471 - 500 CSS related */
  CLSCRS_STAT_CSS_NO_NODE_NUM=471,  /* css api unable to get the node number */
  CLSCRS_STAT_CSS_NO_NODE_NAME=472, /* css could not retrieve the node name */
  CLSCRS_STAT_LONG_CONNECTION_KEY=473,     /* the generated key was too long */
  /* messages 1001 -1028 in use */

  /* messages 2001 - 2020 in use */

  /* messages 2500 - 2800 : new for TrailBlazer */
  CLSCRS_STAT_DISABLED        = 2660, /* cannot touch disabled resources */
  CLSCRS_STAT_CANCELLED       = 2506, /* A scheduled operation was cancelled*/
  CLSCRS_STAT_INTERNAL_ERROR  = 2531, /* Unexpected non-user error */
  CLSCRS_STAT_WRONG_ACTIVE_VERSION
                              = 2839  /* Required Active Version of the cluster
                                         is below the required one */

} CLSCRS_STAT;

/* Return codes for the clscrs_sp, clscrs_splist, clscrs_res, clscrs_reslist 
   funtions */

typedef enum clscrsret
{
  clscrsretSUCC          = 0,    /* Success                                  */
  clscrsretFAIL          = 1,    /* Failure                                  */
  clscrsretNOMEM         = 2,    /* Memory Allocation Failure                */
  clscrsretBADCTX        = 3,    /* Invalid clscrs context                   */
  clscrsretBADARG        = 4,    /* Invalid argument                         */
  clscrsretNONAME        = 5,    /* Given name not found                     */
  clscrsretENDLIST       = 6,    /* End of list reached                      */
  clscrsretNOATTRS       = 7,    /* No attributes set for the resource       */
  clscrsretNOMSG         = 8,    /* No error message available               */
  clscrsretNORES         = 9,    /* Given resource not found                 */
  clscrsretRESEXISTS     = 10,   /* Resource already exists                  */
  clscrsretEMPTY         = 11,   /* List is empty                            */
  clscrsretBADRESNAME    = 12,   /* Invalid resource name                    */
  clscrsretBADNAME       = 13,   /* Invalid argument name                    */
  clscrsretBADENTITY     = 14,   /* Entity mismatch with the call argument   */
  clscrsretLONGNAME      = 15,   /* Attribute name length exceeds limit      */
  /* 16 is available */
  clscrsretBUFFSMALL     = 17    /* Buffer size is small                     */
} clscrsret;

#ifdef __cplusplus
  #define CLSCRSX_ATTR(x) (x)
#else
  #define CLSCRSX_ATTR(x) ((const oratext*)x)
#endif

/* Resource Attributes */

#define clscrs_NAME               CLSCRSX_ATTR("NAME")
#define clscrs_TYPE               CLSCRSX_ATTR("TYPE")
#define clscrs_ACTION_SCRIPT      CLSCRSX_ATTR("ACTION_SCRIPT")
#define clscrs_ACTIVE_PLACEMENT   CLSCRSX_ATTR("ACTIVE_PLACEMENT")
#define clscrs_AUTO_START         CLSCRSX_ATTR("AUTO_START")
#define clscrs_CHECK_INTERVAL     CLSCRSX_ATTR("CHECK_INTERVAL")
#define clscrs_DESCRIPTION        CLSCRSX_ATTR("DESCRIPTION")
#define clscrs_FAILOVER_DELAY     CLSCRSX_ATTR("FAILOVER_DELAY")
#define clscrs_FAILURE_INTERVAL   CLSCRSX_ATTR("FAILURE_INTERVAL")
#define clscrs_FAILURE_THRESHOLD  CLSCRSX_ATTR("FAILURE_THRESHOLD")
#define clscrs_HOSTING_MEMBERS    CLSCRSX_ATTR("HOSTING_MEMBERS")
#define clscrs_OPTIONAL_RESOURCES CLSCRSX_ATTR("OPTIONAL_RESOURCES")
#define clscrs_PLACEMENT          CLSCRSX_ATTR("PLACEMENT")
#define clscrs_REQUIRED_RESOURCES CLSCRSX_ATTR("REQUIRED_RESOURCES")
#define clscrs_ACTIONS            CLSCRSX_ATTR("ACTIONS")
#define clscrs_INSTANCE_FAILOVER  CLSCRSX_ATTR("INSTANCE_FAILOVER")
#define clscrs_RESTART_ATTEMPTS   CLSCRSX_ATTR("RESTART_ATTEMPTS")
#define clscrs_SCRIPT_TIMEOUT     CLSCRSX_ATTR("SCRIPT_TIMEOUT")
#define clscrs_START_TIMEOUT      CLSCRSX_ATTR("START_TIMEOUT")
#define clscrs_STOP_TIMEOUT       CLSCRSX_ATTR("STOP_TIMEOUT")
#define clscrs_DELETE_TIMEOUT     CLSCRSX_ATTR("DELETE_TIMEOUT")
#define clscrs_MODIFY_TIMEOUT     CLSCRSX_ATTR("MODIFY_TIMEOUT")
#define clscrs_ACTION_TIMEOUT     CLSCRSX_ATTR("ACTION_TIMEOUT")
#define clscrs_CHECK_TIMEOUT      CLSCRSX_ATTR("CHECK_TIMEOUT")
#define clscrs_INTERMEDIATE_TIMEOUT CLSCRSX_ATTR("INTERMEDIATE_TIMEOUT")
#define clscrs_OS_CRASH_THRESHOLD CLSCRSX_ATTR("OS_CRASH_THRESHOLD")
#define clscrs_OS_CRASH_UPTIME    CLSCRSX_ATTR("OS_CRASH_UPTIME")
#define clscrs_START_CONCURRENCY  CLSCRSX_ATTR("START_CONCURRENCY")
#define clscrs_STOP_CONCURRENCY   CLSCRSX_ATTR("STOP_CONCURRENCY")
#define clscrs_CLEAN_TIMEOUT      CLSCRSX_ATTR("CLEAN_TIMEOUT")
#define clscrs_UPTIME_THRESHOLD   CLSCRSX_ATTR("UPTIME_THRESHOLD")
#define clscrs_STATE              CLSCRSX_ATTR("STATE")
#define clscrs_TARGET             CLSCRSX_ATTR("TARGET")
#define clscrs_ENABLED            CLSCRSX_ATTR("ENABLED")
#define clscrs_EXISTS             CLSCRSX_ATTR("EXISTS")
#define clscrs_START_ATTEMPTS     CLSCRSX_ATTR("START_ATTEMPTS")
#define clscrs_DEGREE             CLSCRSX_ATTR("DEGREE")
#define clscrs_LOAD               CLSCRSX_ATTR("LOAD")
#define clscrs_LOGGING_LEVEL      CLSCRSX_ATTR("LOGGING_LEVEL")
#define clscrs_START_DEPENDENCIES CLSCRSX_ATTR("START_DEPENDENCIES")
#define clscrs_STOP_DEPENDENCIES  CLSCRSX_ATTR("STOP_DEPENDENCIES")
#define clscrs_DEGREE_ID          CLSCRSX_ATTR("DEGREE_ID")
#define clscrs_ID                 CLSCRSX_ATTR("ID")
#define clscrs_ACL                CLSCRSX_ATTR("ACL")
#define clscrs_CARDINALITY        CLSCRSX_ATTR("CARDINALITY")
#define clscrs_CARDINALITY_ID     CLSCRSX_ATTR("CARDINALITY_ID")
#define clscrs_AGENT_FILENAME     CLSCRSX_ATTR("AGENT_FILENAME")
#define clscrs_DEFAULT_TEMPLATE        CLSCRSX_ATTR("DEFAULT_TEMPLATE")
#define clscrs_STATE_CHANGE_TEMPLATE   CLSCRSX_ATTR("STATE_CHANGE_TEMPLATE")
#define clscrs_PROFILE_CHANGE_TEMPLATE CLSCRSX_ATTR("PROFILE_CHANGE_TEMPLATE")
#define clscrs_ACTION_FAILURE_TEMPLATE CLSCRSX_ATTR("ACTION_FAILURE_TEMPLATE")
#define clscrs_NOT_RESTARTING_TEMPLATE CLSCRSX_ATTR("NOT_RESTARTING_TEMPLATE")
#define clscrs_ALERT_TEMPLATE     CLSCRSX_ATTR("ALERT_TEMPLATE")
#define clscrs_CREATION_SEED      CLSCRSX_ATTR("CREATION_SEED")
#define clscrs_SERVER_POOLS       CLSCRSX_ATTR("SERVER_POOLS")
#define clscrs_SERVER_CATEGORY    CLSCRSX_ATTR("SERVER_CATEGORY")
#define clscrs_USER_WORKLOAD      CLSCRSX_ATTR("USER_WORKLOAD")
#define clscrs_ALIAS_NAME         CLSCRSX_ATTR("ALIAS_NAME")
#define clscrs_LAST_SERVER        CLSCRSX_ATTR("LAST_SERVER")
#define clscrs_TARGET_SERVER      CLSCRSX_ATTR("TARGET_SERVER")
#define clscrs_OFFLINE_CHECK_INTERVAL  \
                                  CLSCRSX_ATTR("OFFLINE_CHECK_INTERVAL")
#define clscrs_STATE_DETAILS      CLSCRSX_ATTR("STATE_DETAILS")    
#define clscrs_INCARNATION        CLSCRSX_ATTR("INCARNATION")
#define clscrs_CURRENT_RCOUNT     CLSCRSX_ATTR("RESTART_COUNT")
#define clscrs_FAILURE_COUNT      CLSCRSX_ATTR("FAILURE_COUNT")
#define clscrs_FAILURE_HISTORY    CLSCRSX_ATTR("FAILURE_HISTORY")
#define clscrs_LAST_RESTART       CLSCRSX_ATTR("LAST_RESTART")
#define clscrs_LAST_STATE_CHANGE  CLSCRSX_ATTR("LAST_STATE_CHANGE")
#define clscrs_INTERNAL_STATE     CLSCRSX_ATTR("INTERNAL_STATE")
#define clscrs_RELOCATE_BY_DEPENDENCY \
                                  CLSCRSX_ATTR("RELOCATE_BY_DEPENDENCY")
#define clscrs_USE_STICKINESS     CLSCRSX_ATTR("USE_STICKINESS")
#define clscrs_INSTANCE_COUNT     CLSCRSX_ATTR("INSTANCE_COUNT")

/* server pool attributes */
#define clscrs_SERVER_NAMES       CLSCRSX_ATTR("SERVER_NAMES")
#define clscrs_MIN_SIZE           CLSCRSX_ATTR("MIN_SIZE")
#define clscrs_MAX_SIZE           CLSCRSX_ATTR("MAX_SIZE")
#define clscrs_IMPORTANCE         CLSCRSX_ATTR("IMPORTANCE")
#define clscrs_PARENT_POOLS       CLSCRSX_ATTR("PARENT_POOLS")
#define clscrs_EXCLUSIVE_POOLS    CLSCRSX_ATTR("EXCLUSIVE_POOLS")
#define clscrs_ACTIVE_SERVERS     CLSCRSX_ATTR("ACTIVE_SERVERS")

/* config policy attribute */
#define clscrs_POLICIES              CLSCRSX_ATTR("POLICIES")
#define clscrs_LAST_ACTIVATED_POLICY CLSCRSX_ATTR("LAST_ACTIVATED_POLICY")
#define clscrs_SERVER_POOL_NAMES     CLSCRSX_ATTR("SERVER_POOL_NAMES")

/* server category specifics */
#define clscrs_EXPRESSION            CLSCRSX_ATTR("EXPRESSION")

/* server attributes */
#define clscrs_SERVER_NAME          CLSCRSX_ATTR("NAME")
#define clscrs_ACTIVE_POOLS         CLSCRSX_ATTR("ACTIVE_POOLS")
#define clscrs_SERVER_STATE         clscrs_STATE
#define clscrs_SERVER_STATE_DETAILS clscrs_STATE_DETAILS
#define clscrs_MEMORY_SIZE          CLSCRSX_ATTR("MEMORY_SIZE")
#define clscrs_CPU_COUNT            CLSCRSX_ATTR("CPU_COUNT")
#define clscrs_CPU_CLOCK_RATE       CLSCRSX_ATTR("CPU_CLOCK_RATE")
#define clscrs_CPU_HYPERTHREADING   CLSCRSX_ATTR("CPU_HYPERTHREADING")   
#define clscrs_ACTIVE_CSS_ROLE      CLSCRSX_ATTR("ACTIVE_CSS_ROLE")
#define clscrs_CONFIGURED_CSS_ROLE  CLSCRSX_ATTR("CONFIGURED_CSS_ROLE")
#define clscrs_DEPLOYMENT           CLSCRSX_ATTR("DEPLOYMENT")
#define clscrs_WORKLOAD_VALUE       CLSCRSX_ATTR("WORKLOAD_VALUE")
#define clscrs_CPU_EQUIVALENCY      CLSCRSX_ATTR("CPU_EQUIVALENCY")
#define clscrs_RESOURCE_USE_ENABLED CLSCRSX_ATTR("RESOURCE_USE_ENABLED")
#define clscrs_SERVER_LABEL         CLSCRSX_ATTR("SERVER_LABEL")
#define clscrs_PHYSICAL_HOSTNAME    CLSCRSX_ATTR("PHYSICAL_HOSTNAME")

/* resource type attributes */
#define clscrs_TYPE_NAME          CLSCRSX_ATTR("TYPE_NAME")
#define clscrs_BASE_TYPE          CLSCRSX_ATTR("BASE_TYPE")
#define clscrs_TYPE_ACL           CLSCRSX_ATTR("TYPE_ACL")

/* generic application type attributes */
#define clscrs_START_PROGRAM       CLSCRSX_ATTR("START_PROGRAM")
#define clscrs_STOP_PROGRAM        CLSCRSX_ATTR("STOP_PROGRAM")
#define clscrs_CLEAN_PROGRAM       CLSCRSX_ATTR("CLEAN_PROGRAM")
#define clscrs_CHECK_PROGRAMS      CLSCRSX_ATTR("CHECK_PROGRAMS")
#define clscrs_PID_FILES           CLSCRSX_ATTR("PID_FILES")
#define clscrs_EXECUTABLE_NAMES    CLSCRSX_ATTR("EXECUTABLE_NAMES")
#define clscrs_ENVIRONMENT_VARS    CLSCRSX_ATTR("ENVIRONMENT_VARS")
#define clscrs_ENVIRONMENT_FILE    CLSCRSX_ATTR("ENVIRONMENT_FILE")
#define clscrs_PID_LIST            CLSCRSX_ATTR("TEMP_PID_LIST")
#define clscrs_SEND_OUTPUT_ALWAYS  CLSCRSX_ATTR("SEND_OUTPUT_ALWAYS")

/* agent framework attributes */
#define clscrs_AGENT_NAME         CLSCRSX_ATTR("AGENT_NAME")
#define clscrs_CHECK_TIMEOUT      CLSCRSX_ATTR("CHECK_TIMEOUT")
#define clscrs_CLEAN_TIMEOUT      CLSCRSX_ATTR("CLEAN_TIMEOUT")
#define clscrs_CREATION_SEED      CLSCRSX_ATTR("CREATION_SEED")
#define clscrs_OFFLINE_TIMEOUT    CLSCRSX_ATTR("OFFLINE_TIMEOUT")
#define clscrs_ONLINE_TIMEOUT     CLSCRSX_ATTR("ONLINE_TIMEOUT")
#define clscrs_PATH_NAME          CLSCRSX_ATTR("PATH_NAME")

/* special values for resource profiles */
#define clscrs_SV_CRS_HOME           CLSCRSX_ATTR("CRS_HOME")
#define clscrs_SV_CSS_NODENAME       CLSCRSX_ATTR("CRS_CSS_NODENAME")
#define clscrs_SV_CSS_NODENUMBER     CLSCRSX_ATTR("CRS_CSS_NODENUMBER")
#define clscrs_SV_CSS_NODENUMBER_PLUS1 \
                                     CLSCRSX_ATTR("CRS_CSS_NODENUMBER_PLUS1")
#define clscrs_SV_CSS_NODENAME_UPPER CLSCRSX_ATTR("CRS_CSS_NODENAME_UPPER_CASE")
#define clscrs_SV_CSS_NODENAME_LOWER CLSCRSX_ATTR("CRS_CSS_NODENAME_LOWER_CASE")
#define clscrs_SV_CLUSTER_SIZE       CLSCRSX_ATTR("CRS_CLUSTER_SIZE")
#define clscrs_SV_HUB_SIZE           CLSCRSX_ATTR("CRS_HUB_SIZE")
#define clscrs_SV_CLUSTER_NAME       CLSCRSX_ATTR("CRS_CLUSTER_NAME")
#define clscrs_SV_SERVERPOOL_SIZE    CLSCRSX_ATTR("CRS_SERVER_POOL_SIZE")
#define clscrs_SV_ONLINE_CSS_NODENAMES \
                                     CLSCRSX_ATTR("CRS_ONLINE_CSS_NODENAMES")
#define clscrs_SV_EXE_SUFFIX         CLSCRSX_ATTR("CRS_EXE_SUFFIX")
#define clscrs_SV_SCRIPT_SUFFIX      CLSCRSX_ATTR("CRS_SCRIPT_SUFFIX")
#define clscrs_SV_ORIGINAL_REASON    CLSCRSX_ATTR("CRS_ORIGINAL_REASON")

/* resource STATE values */
#define clscrs_STATE_ONLINE          CLSCRSX_ATTR("ONLINE")
#define clscrs_STATE_OFFLINE         CLSCRSX_ATTR("OFFLINE")
#define clscrs_STATE_INTERMEDIATE    CLSCRSX_ATTR("INTERMEDIATE")
#define clscrs_STATE_UNKNOWN         CLSCRSX_ATTR("UNKNOWN")


/* resource INTERNAL_STATE values */
#define clscrs_ISTATE_STARTING       CLSCRSX_ATTR("STARTING")
#define clscrs_ISTATE_STOPPING       CLSCRSX_ATTR("STOPPING")
#define clscrs_ISTATE_CLEANING       CLSCRSX_ATTR("CLEANING")
#define clscrs_ISTATE_CHECKING       CLSCRSX_ATTR("CHECKING")
#define clscrs_ISTATE_STABLE         CLSCRSX_ATTR("STABLE")

/* resource TARGET STATE values */
#define clscrs_TARGET_STATE_ONLINE   CLSCRSX_ATTR("ONLINE")
#define clscrs_TARGET_STATE_OFFLINE  CLSCRSX_ATTR("OFFLINE")

/* built-in resource types */
#define clscrs_APPLICATION_TYPE_NAME          CLSCRSX_ATTR("application")
#define clscrs_RESOURCE_TYPE_NAME             CLSCRSX_ATTR("resource")
#define clscrs_CLUSTER_RESOURCE_TYPE_NAME     CLSCRSX_ATTR("cluster_resource")
#define clscrs_LOCAL_RESOURCE_TYPE_NAME       CLSCRSX_ATTR("local_resource")
#define clscrs_GENERIC_APPLICATION_TYPE_NAME  CLSCRSX_ATTR("generic_application")

/* built-in server pool names */
#define clscrs_FREE_SERVER_POOL_NAME     CLSCRSX_ATTR("Free")
#define clscrs_GENERIC_SERVER_POOL_NAME  CLSCRSX_ATTR("Generic")

/* Values for css role */
#define clscrs_CSS_ROLE_RIM              CLSCRSX_ATTR("leaf")
#define clscrs_CSS_ROLE_HUB              CLSCRSX_ATTR("hub")
#define clscrs_CSS_ROLE_AUTO             CLSCRSX_ATTR("auto")
#define clscrs_TARGET_ROLE               CLSCRSX_ATTR("TARGET_ROLE")

/* Values for DEPLOYMENT */
#define clscrs_DEPLOYMENT_OTHER          CLSCRSX_ATTR("other")
#define clscrs_DEPLOYMENT_DBVM           CLSCRSX_ATTR("dbvm")
#define clscrs_DEPLOYMENT_GIVM           CLSCRSX_ATTR("givm")

/* built-in config policy names */
#define clscrs_CURRENT_CONFIG_POLICY_NAME CLSCRSX_ATTR("Current")

/* reserved values for resource profiles */
#define clscrs_ANY_SERVER_POOL        CLSCRSX_ATTR("*")
#define clscrs_BALANCED_PLACEMENT     CLSCRSX_ATTR("balanced")
#define clscrs_FAVORED_PLACEMENT      CLSCRSX_ATTR("favored")
#define clscrs_RESTRICTED_PLACEMENT   CLSCRSX_ATTR("restricted")

/* values for user/group identification in ACTIONS attribute */
#define clscrs_ACTIONS_USER           CLSCRSX_ATTR("user:")
#define clscrs_ACTIONS_GROUP          CLSCRSX_ATTR("group:")

/* values for server state: state values must be kept in sync. with 
 * ServerState enum in opsm/jsrc/oracle/cluster/server/Server.java */
#define clscrs_SS_ONLINE              CLSCRSX_ATTR("ONLINE")
#define clscrs_SS_OFFLINE             CLSCRSX_ATTR("OFFLINE")
#define clscrs_SS_VISIBLE             CLSCRSX_ATTR("VISIBLE")
#define clscrs_SS_JOINING             CLSCRSX_ATTR("JOINING")
#define clscrs_SS_LEAVING             CLSCRSX_ATTR("LEAVING")
#define clscrs_SS_RECONFIGURING       CLSCRSX_ATTR("RECONFIGURING")

/* values for SERVER_STATE_DETAILS */
#define clscrs_SSD_STARTING_RESOURCES CLSCRSX_ATTR("STARTING RESOURCES")
#define clscrs_SSD_CHECKING_RESOURCES CLSCRSX_ATTR("CHECKING RESOURCES")
#define clscrs_SSD_AUTOSTARTING_RESOURCES \
                                      CLSCRSX_ATTR("AUTOSTARTING RESOURCES")
#define clscrs_SSD_STOPPING_RESOURCES CLSCRSX_ATTR("STOPPING RESOURCES")
#define clscrs_SSD_RECONFIG_FAILED    CLSCRSX_ATTR("RECONFIG FAILED")
#define clscrs_SSD_AUTOSTART_QUEUED   CLSCRSX_ATTR("AUTOSTART QUEUED")

/* NLS_LANG settings in client requests */
#define clscrs_CLIENT_NLS_LANG        CLSCRSX_ATTR("CRS_NLS_LANG")

/* Volume resource attributes */
#define clscrs_VOLUME_DEVICE  CLSCRSX_ATTR("VOLUME_DEVICE")

/* filter constants */
#define clscrs_EQ_COMPARATOR             CLSCRS0_ATTR("==")
#define clscrs_NE_COMPARATOR             CLSCRS0_ATTR("!=")
#define clscrs_GT_COMPARATOR             CLSCRS0_ATTR(">")
#define clscrs_LT_COMPARATOR             CLSCRS0_ATTR("<")
#define clscrs_CO_COMPARATOR             CLSCRS0_ATTR("co")
#define clscrs_ST_COMPARATOR             CLSCRS0_ATTR("st")
#define clscrs_EN_COMPARATOR             CLSCRS0_ATTR("en")
#define clscrs_NC_COMPARATOR             CLSCRS0_ATTR("nc")
#define clscrs_NCI_COMPARATOR            CLSCRS0_ATTR("nci")
#define clscrs_COI_COMPARATOR            CLSCRS0_ATTR("coi")
#define clscrs_EQI_COMPARATOR            CLSCRS0_ATTR("eqi")
#define clscrs_AND_OPERATOR              CLSCRS0_ATTR("&&")
#define clscrs_OR_OPERATOR               CLSCRS0_ATTR("||")
#define clscrs_LEFT_BRACKET              CLSCRS0_ATTR("(")
#define clscrs_RIGHT_BRACKET             CLSCRS0_ATTR(")")

/* name limits */
#define CLSCRS_ENTITYNAME_MAXLEN  255   /* entity name maximum length */
#define CLSCRS_ATTRNAME_MAXLEN 255      /* attribute name maximum length */

/* Maximum length of the key value in the ocr */
#define MAXOCRLEN (PROCR_MAX_VALUESIZE - 1) 
                                                  
/* Delimiter character for attributes */
#define CLSCRS_ATTRIBUTE_DELIMITER  '\001'

/* ACL roles and separators */
#define clscrs_ROLE_OTHER          CLSCRS0_ATTR("other")
#define clscrs_ROLE_OWNER          CLSCRS0_ATTR("owner")
#define clscrs_ROLE_USER           CLSCRS0_ATTR("user")
#define clscrs_PERM_SEPARATOR      CLSCRS0_ATTR(":")
#define clscrs_ACL_SEPARATOR       CLSCRS0_ATTR(",")

/* Structure containing clscrs "global" context */
typedef struct clscrs_ctx clscrs_ctx;

/* Structure representing a resource.  
   NOTE: This has been DEPRECATED (after 11.2). 
         Please use clscrs_crsentity.*/
typedef struct clscrs_res clscrs_res;

/* Structure representing a CRS entity */
typedef struct clscrs_res clscrs_crsentity;

/* Structure for a list of resources 
   NOTE: This has been DEPRECATED (after 11.2). 
         Please use clscrs_crsentitylist.*/
typedef struct clscrs_reslist clscrs_reslist;

/* Structure representing a list of CRS entities */
typedef struct clscrs_reslist clscrs_crsentitylist;

/* Structure representing a stringpair */
typedef struct clscrs_sp clscrs_sp;

/* Structure for a list of stringpairs */
typedef struct clscrs_splist clscrs_splist;

/* Structure for representing a single what-if action, with parameters */
typedef struct clscrs_action clscrs_action;

/* Structure for representing a sequence of what-if actions */
typedef struct clscrs_actionlist clscrs_actionlist;

/* Structure for passing enviornment variables to start and stop operations */
typedef struct clscrs_env_item *clscrs_env;

/* Structure for representing a filter*/
typedef struct clscrs_splist clscrs_filter;

/* Structure for representing a clscrs query*/
typedef struct clscrs_query clscrs_query;

/* Structure for representing a clscrs querylist*/
typedef struct clscrs_querylist clscrs_querylist;

/* enum to specify the list of attributes queried in the clscrs_stat() api */
typedef enum 
{
  clscrs_attrs_none     = 1,
  clscrs_attrs_all      = 2,
  clscrs_attrs_short    = 3,
  clscrs_attrs_state    = 4
} clscrs_attr_grp;

/* enum to create a comparison/simple filter */
typedef enum 
{
  clscrs_comparator_invalid = 0,   /* invalid comparator */
  clscrs_comparator_eq      = 1,   /* equals */
  clscrs_comparator_ne      = 2,   /* not equals */
  clscrs_comparator_gt      = 3,   /* greater than */
  clscrs_comparator_lt      = 4,   /* less than */
  clscrs_comparator_co      = 5,   /* contains */
  clscrs_comparator_starts  = 6,   /* starts with */
  clscrs_comparator_ends    = 7,   /* ends with */
  clscrs_comparator_nc      = 8,   /* not contains */
  clscrs_comparator_nci     = 9,   /* not contains case-insensitive */
  clscrs_comparator_coi     = 10,  /* contains case-insensitive */
  clscrs_comparator_eqi     = 11   /* equals case-insensitive */
} clscrs_comparator;

/* enum to create an expression filter */
typedef enum 
{
  clscrs_operator_invalid = 0,  /* invalid operator */
  clscrs_operator_and     = 1,  /* AND operator */
  clscrs_operator_or      = 2   /* OR  operator */
} clscrs_operator;

/* enum to specify the entity type */
typedef enum 
{
  clscrs_entity_res              = 1,  /* resource */
  clscrs_entity_restype          = 2,  /* resource type */
  clscrs_entity_serverpool       = 3,  /* server pool */
  clscrs_entity_server           = 4,  /* server */
  clscrs_entity_resinst          = 5,  /* resource instances */
  clscrs_entity_configpolicy     = 6,  /* configuration policy */ 
  clscrs_entity_configpolicy_set = 7,  /* configuration policy set */ 
  clscrs_entity_servercategory   = 8   /* server category */ 

} clscrs_entity;

/* new typedef to rename clscr_entity as clscrs_entity_type */
typedef clscrs_entity clscrs_entity_type;

/* enum to identify the message type in the user callback function */
typedef enum 
{
  clscrs_msgtype_error     = 1,  /* error message */
  clscrs_msgtype_warning   = 2,  /* warning message */
  clscrs_msgtype_progress  = 3   /* progress message */
}clscrs_msgtype;

/* enum to identify the various types of possible what-if actions */
typedef enum
{
  clscrs_actiontype_resstate  = 1, /* State of a resource changes */
  clscrs_actiontype_srvmove   = 2, /* Server moves to a different pool */
  clscrs_actiontype_newgroup  = 3, /* Creation of a new stage group */
  clscrs_actiontype_errorcase = 4  /* Handles error cases */
} clscrs_actiontype;


/* Environment operation status domain. */

typedef enum
{
  CLSCRS_ENV_SUCCESS=0,
  CLSCRS_ENV_SYNTAX=101,
  CLSCRS_ENV_VALUE=102,
  CLSCRS_ENV_ATTR_NOTFOUND=103,
  CLSCRS_ENV_NULL=104,
  CLSCRS_ENV_FAILURE=105
} CLSCRS_ENVCODE;

/* Flags passed to the different api */
typedef enum
{
  /* default flag */
  CLSCRS_FLAG_NONE=0,
  /* flag for asynchronous execution of the api. The call will return before the 
     request is processed completely. If this flag is not specified, the api
     call will block until the request is completely processed */
  CLSCRS_FLAG_ASYNC=0x01,
  /* flag for the 'forceful' flavor of the action. What this means to the
   * particular API, if anything, should be discussed in the
   * description of that API function*/
  CLSCRS_FLAG_FORCE=0x02,
  /* flag to enable tracing via the callback function */
  CLSCRS_FLAG_TRACE=0x04,
  /* flag to enable multithreading. This flag will enable the use of the clscrs
     context in mutliple threads. The context should be initialized while the 
     application is single-threaded. */
  CLSCRS_FLAG_USETHREADS=0x08,
  /* flag passed to register api for updating the attributes of an already 
     registered resource */
  CLSCRS_FLAG_REG_UPDATE=0x10,
  /* 
     DEPRECATED: Use *_T_HA_* flags instead

     flag passed to start and stop api to indicate the start/stop event rather
     than a request to run the start/stop actions. Practically, this flag
     tells the server to override hands-of-UNKNOWN resources logic and will 
     allow invocation of START on a resoure that's in UNKNOWN state */
  CLSCRS_FLAG_EVENT=0x20,
   /* flag that specifies the api request to be queued at the server if the 
     request could not be processed immediately because one or more of the
     affected entities is locked for another opertion; if set, this will
     make the server queue up the request and execute it whenever the
     locking conflict clears. */
  CLSCRS_FLAG_QUEUE=0x40,
  /* flag for ohasd. flag that instructs the api to talk to ohasd instead
     of crsd.  */
  CLSCRS_FLAG_CLSINIT=0x80,
  /* flag indicating all resources to the start, stop, unregister, relocate 
     operations api */
  CLSCRS_FLAG_ALL=0x100,
  /* flag passed to register api for purging the per-X attributes of an already 
     registered resource; please note that this flag will only have effect if
     used in conjunction with the CLSCRS_FLAG_REG_UPDATE flag */
  CLSCRS_FLAG_REG_PURGE=0x200,
  /* flag that is used for stopping resources to let crsd know that it
   * should not change the targets of dependent resources to offline */
  CLSCRS_FLAG_KEEP_DEP_TARGET = 0x400,
  /* Used in Transparent HA (start2/stop2 APIs) to instruct crsd to prepare for
   * an external management interface's acting on a resource. When the
   * call comes back, if successful, the interface then is expected to
   * start/stop the resource and call back with the other flag
   * (CLSCRS_DLAG_T_HA_FINISH) */
  CLSCRS_FLAG_T_HA_PREPARE = 0x800,
  /* Used in Transparent HA in conjunction with
   * CLSCRS_FLAG_T_HA_PREPARE. Must be used in the second call to crsd
   * (start2/stop2 APIs) when the action (start or stop) has finished. 
   * Note that clients must indicate indicate resource instances and
   * never resources with this flag, as crsd must know which
   * instance(s) the invocation of the API applies */
  CLSCRS_FLAG_T_HA_FINISH = 0x1000,

  /*
   * Used when rolling upgrade process is going on. 
   * While rolling upgrade going on crs does not accept any new 
   * commands. Unless this special flag is mentioned in APIs.
   */
  CLSCRS_FLAG_UPGRADE = 0x2000,

  /*
   * Available for use in clscrs_register_serverpool() to request that
   * a "what if this pool were created" or "what if this pool where
   * modified" - kind of evaluation be performed. If specified, will make the
   * API return errors specifying which resources will be affected,
   * if any; however, the pool will not be created/modified - this 
   * is just an evalution of what would be affected.  
   * Errors will be returned via the progress callback. 
   */
  CLSCRS_FLAG_EVAL_ONLY = 0x4000,

   /*
   * It's possible that CRS is not ready to process any
   * requests, in which case it queues them and processes them as soon
   * as it can. However, the amount of time the request remains queued
   * is not determinic. Thus some clients may wish to get back an
   * error as opposed to being blocked for an uncontrolled amount of
   * time; such clients need to supply this flag. If CRS is currently
   * not able to process the request, the client will get a connection
   * error immediately.
   * Note that there's a very subtle difference between this flag
   * and CLSCRS_FLAG_ASYNC. The latter means: deliver the request and
   * return immediately. This flag means: deliver the request and
   * either wait for it to complete OR, if CRS is not processing
   * requests in principle, return back a connection error as if CRS
   * stack was down completely.
   * This flag may come handy when a resource is being acted on from
   * an interface other than CRS, because if CRS is were used to
   * manage the resource, it would not issue anything until it was
   * ready in the first place.
   *
   * see also CLSCRS_STATFLAG_NOWAIT
   */
   CLSCRS_FLAG_NOWAIT = 0x8000,


   /*
    * Available for use for start/stop resource APIs to
    * instruct crsd to skip resource state restoration of the
    * resources whose state was affected. That procedure is usually
    * attempted unless this flag was used
    */  
    CLSCRS_FLAG_NO_STATE_RESTORATION = 0x00010000,


   /*
    * Used in whatif calls to request verbose information.
    * When this is specified, additional information regarding
    * the whatif actions is provided. Currently, we add Initial Server, 
    * Initial State and Initial Target values to the resource actions.
    * This is primarily used in CRSCTL Cluster Admin views.
    */  
    CLSCRS_FLAG_WHATIF_VERBOSE = 0x00020000,


   /*
    * When specified while updating a policy set, this will remove 
    * server pools from the policy set such that they are
    * not actually deleted from the system but rather become "independent" 
    * (not managed by the policy set).
    */ 
    CLSCRS_FLAG_KEEP_POOLS = 0x00040000
  
} CLSCRS_FLAG;

/* Flags passed to the clscrs_stat2() api */
typedef enum
{
  CLSCRS_STATFLAG_NONE=0,
  /* DEPRECATED: use clscrs_entity_resinst as the entity and, optionally,
   *             a filter instead
   * flag to query the resource instances of a resource, 
     valid for entity clscrs_entity_res only.*/
  CLSCRS_STATFLAG_RESINST=0x01,
  /* flag to query the candidate servers to relocate a resource,
     valid for entity clscrs_entity_res only */
  CLSCRS_STATFLAG_RELSERVERS=0x02,
  /* flag to get the complete attribute set of a type, including its
     base types */
  CLSCRS_STATFLAG_FULLTYPE=0x04,

  /*
   * Used when rolling upgrade process is going on. 
   * While rolling upgrade going on crs does not accept any new 
   * commands. Unless this special flag is mentioned in APIs.
   */
  CLSCRS_STATFLAG_UPGRADE=0x08,
  /* Flag to get the evaluated values for special attributes 
   * Note that this will work only for Resource Instances.*/
  CLSCRS_STATFLAG_EVAL=0x10,

  /*
   * Normally, stat2 requests are processed immediately as soon as
   * they are received because they do not require any locks.
   * However, it's possible that CRS is not ready to process any
   * requests, in which case it queues them and processes them as soon
   * as it can. However, the amount of time the request remains queued
   * is not determinic. Thus some clients may wish to get back an
   * error as opposed to being blocked for an uncontrolled amount of
   * time; such clients need to supply this flag. If CRS is currently
   * not able to process the request, the client will get a connection
   * error immediately.
   */
  CLSCRS_STATFLAG_NOWAIT = 0x20,
   
  /* 
   * Used to to query the servers that match a particular
   * server category. If the category name is empty, an empty server
   * list is returned.
   */
  CLSCRS_STATFLAG_SERVERBYCATEGORY=0x40,

  /* 
   * Used to query the server categories that match a particular server
   */
  CLSCRS_STATFLAG_CATEGORYBYSERVER=0x80

} CLSCRS_STATFLAG;

/* Flags used for attributes. Note that only some of these can be used
  when creating user-defined types. See the API description to find out
  which apply */
typedef enum
{
  /* Once created, the attribute value may not be modified */
  CLSCRS_TYPEFLAG_READONLY=0x01,  

  /* The attribute is a part of the resource's configuration vs. state */
  CLSCRS_TYPEFLAG_CONFIG=0x02,    

  /* A value for this attribute is required at resource creation time */
  CLSCRS_TYPEFLAG_REQUIRED=0x04,

  /* The attribute's value takes effect immediately when modified, without having to
   * restart the resource */
  CLSCRS_TYPEFLAG_HOTMOD=0x08

} CLSCRS_TYPEFLAG;

typedef enum clscrs_dtype
{
  clscrs_dtype_INT      = 0,    /* integer data type */
  clscrs_dtype_STRING   = 1     /* string data type */
} clscrs_dtype;

/* these macros are applicable only in 10gR2 - DEPRECATED */
#define CLSCRS_ATTRS_NONE(ctx)     \
      (clscrs_get_fixed_attrlist(ctx, clscrs_attrs_none))
#define CLSCRS_ATTRS_ALL(ctx)     \
      (clscrs_get_fixed_attrlist(ctx, clscrs_attrs_all))
#define CLSCRS_ATTRS_STATE(ctx)     \
      (clscrs_get_fixed_attrlist(ctx, clscrs_attrs_state))
#define CLSCRS_ATTRS_SHORT(ctx)     \
      (clscrs_get_fixed_attrlist(ctx, clscrs_attrs_short))

/*---------------------------------------------------------------------------
                     PRIVATE TYPES AND CONSTANTS
  ---------------------------------------------------------------------------*/


/*---------------------------------------------------------------------------
                           EXPORT FUNCTIONS
  ---------------------------------------------------------------------------*/

/* ---------------------------- clscrs_msgf ---------------------------- */
/*
  NAME
    clscrs_msgf

  DESCRIPTION
    User function called to deliver output text from CRS operations.

  PARAMETERS
    usrp    -- user supplied pointer, probably containing context of the call.
    msg     -- output text
    msglen  -- length of the output text.

  RETURNS
    none
*/

typedef  void (*clscrs_msgf)(void *usrp, const oratext *msg, sword msglen);

/* ---------------------------- clscrs_msgf2 ---------------------------- */
/*
  NAME
    clscrs_msgf2

  DESCRIPTION
    User function called to deliver output text from CRS operations.

  PARAMETERS
    usrp    -- user supplied pointer, probably containing context of the call.
    id      -- The identifier of the entity to which the msg corresponds to
    msg     -- output text
    msgtype -- type of the message - progress, warning or error

  RETURNS
    none
*/

typedef  void (*clscrs_msgf2)(void *usrp, const oratext *id, const oratext *msg, 
                              clscrs_msgtype msgtype);

/* ------------------------- clscrs_init_crs ---------------------------- */
/*
  NAME
    clscrs_init_crs

  DESCRIPTION
    Initializes a context for communications with CRS. 

    Note that the function will initialize clsc in either threaded or 
    non-threaded manner, consistent with the use of the 
    CLSCRS_FLAG_USETHREADS flag. It is assumed that the invocation of this 
    function is properly ordered by the client subject to the initialization
    requirements of the process.
    
    To enable debug traces, the callback function and the flag CLSCRS_FLAG_TRACE
    must be specified.

  PARAMETERS
    ctx    [OUT]    -- context to initialize, function allocates storage.
    errf   [IN]     -- callback func for the error info, or NULL
    errCtx [IN]     -- Context for the error callback func, or NULL
    flags  [IN]     -- CLSCRS_FLAG_TRACE, to enable tracing of lower layers

  RETURNS
    CLSCRS_STAT_SUCCESS - on success
    CLSCRS_STAT_FAILURE - on any failure

*/

CLSCRS_STAT clscrs_init_crs(clscrs_ctx **ctx, clscrs_msgf errf, void *errCtx, 
                            ub4 flags);


/* ---------------------------- clscrs_term_crs ---------------------------- */
/*
  NAME
    clscrs_term_crs

  DESCRIPTION
    Releases a context for communications with CRS.

  PARAMETERS
    ctx [IN]    -- context previously initialized with clscrs_init_crs.

  RETURNS
    CLSCRS_STAT_SUCCESS - on success
*/

CLSCRS_STAT clscrs_term_crs(clscrs_ctx **ctx);


/* ------------------------ clscrs_env_create ---------------------------- */
/*
  NAME
    clscrs_env_create

  DESCRIPTION
    Creates a call environment describing parameters to CRS calls.
    The memory is owned by clscrs, release env clscrs_env_delete

  PARAMETERS
    ctx [IN]   -- input context
    env [OUT]  -- env pointer.

  RETURNS
    CLSCRS_STAT_SUCCESS - on success
    CLSCRS_STAT_FAILURE - on any failure
*/

CLSCRS_STAT clscrs_env_create( clscrs_ctx *ctx, clscrs_env *env );


/* ---------------------------- clscrs_env_set ---------------------------- */
/*
  NAME
    clscrs_env_set

  DESCRIPTION
    Sets an attribute/value argument in an environment for CRS calls; val
    may be null.
    
  PARAMETERS
    env  [IN]   -- previously created environment
    attr [IN]   -- attribute name
    val  [IN]   -- attribute value, may be null pointer.

  RETURNS
    CLSCRS_ENV_SUCCESS - on success
    CLSCRS_ENV_FAILURE - on any failure
    CLSCRS_ENV_NULL    - if 'env' argument is NULL
*/

CLSCRS_ENVCODE 
clscrs_env_set( clscrs_env env, const  oratext *attr, const oratext *val);


/* -------------------------- clscrs_env_delete ---------------------------- */
/*
  NAME
    clscrs_env_delete

  DESCRIPTION
    Deletes an environment for CRS calls.   The environment must
    be created again before use.  You may not delete it and then
    set new parameters.
    
  PARAMETERS
    env [IN]    -- the environment whose contents are deleted.

  RETURNS
    CLSCRS_ENV_SUCCESS - on success
    CLSCRS_ENV_NULL    - if 'env' argument is NULL
*/

CLSCRS_ENVCODE clscrs_env_delete( clscrs_env env );


/* ---------------------------- clscrs_env_format ------------------------- */
/*
  NAME
    clscrs_env_format

  DESCRIPTION
    Drive a callback with formatted environment lines.

  PARAMETERS
        env     -- env to dump
        msgf    -- callback function
        msgp    -- argument to msgp, uninterpreted.

  RETURNS
    None
    
  MODIFIED   (MM/DD/YY)
    dbrower   08/26/02 -  created.
*/

void clscrs_env_format(clscrs_env env, clscrs_msgf msgf, void *msgp);

/* ---------------------------- clscrs_getnodename ------------------------- */
/*
  NAME
    clscrs_getnodename

  DESCRIPTION
    Function for getting the nodename; if in a configured cluster, this is
    the name as known to the clusterware.  If not in a cluster, the name
    as known through other OSD means.

    Getting a node name may be expensive in the first call,
    especially if CLSCRS was not initialized with a pre-existing
    CSS context.

  PARAMETERS
    ctx        [IN]  clscrs context 
    nodename   [OUT] name of the local node

  RETURNS
    CLSCRS_STAT_CSS_NO_NODE_NUM  - when css api is unable to get the node number
    CLSCRS_STAT_SUCCESS          - on success
*/

CLSCRS_STAT
clscrs_getnodename(clscrs_ctx *ctx, oratext *nodename);

/* ---------------------------- clscrs_get_crshome ------------------------- */
/*
  NAME
    clscrs_get_crshome

  DESCRIPTION

    This function returns the path for ORA_CRS_HOME.

  PARAMETERS
    ctx       [IN]   clscrs context
    crshome   [OUT]  path for ORA_CRS_HOME
    bufsize   [IN]   size of the buffer for crshome

  RETURNS
    CLSCRS_STAT_SUCCESS - on success
    CLSCRS_STAT_FAILURE - on any failure
*/

CLSCRS_STAT 
clscrs_get_crshome(clscrs_ctx *ctx, oratext *crshome, size_t *bufsize);

/* --------------------- clscrs_register_resource -------------------------- */
/*
  NAME
    clscrs_register_resource DEPRECATED

  DESCRIPTION
    Register the resources in the input resource list. The attributes for the 
    resources are encapsulated in the input resource list. The output op_status
    list contains the results of the register operation for each resource, and 
    contains no valid resource attributes. 

    The caller must create and populate the in_reslist and must create the 
    op_status list. Both the lists must be destroyed by the caller. The 
    op_status list cannot be reused with another api call. It must be created 
    and destroyed for each api call.

    One or more attributes of an already registered resource can be modified by
    passing the CLSCRS_FLAG_REG_UPDATE flag. The flags apply to all resources 
    in the input reslist.
    
  PARAMETERS

    in_reslist [IN]  -  list of resources to be registered
    flags      [IN]  -  CLSCRS_FLAG_REG_UPDATE or 0.
    op_status  [OUT] -  resource list holding the status of the register
                        operation for each resource.

  RETURNS
    CLSCRS_STAT_SUCCESS    - if all input resources are successfully registered
    CLSCRS_STAT_FAILURE    - if atleast one resource could not be registered
    CLSCRS_STAT_CONNECTION - if there is a communication error with the crsd */

CLSCRS_STAT
clscrs_register_resource(clscrs_reslist *in_reslist, uword flags, 
                         clscrs_reslist *op_status);

/* --------------------- clscrs_register_resource2 -------------------------*/
/*
  NAME
    clscrs_register_resource2

  DESCRIPTION
    Register the resources in the input resource list. The attributes for the 
    resources are encapsulated in the input resource list. The output op_status
    list contains the results of the register operation for each resource, and 
    contains no valid resource attributes. 

    The caller must create and populate the in_reslist and must create the 
    op_status list. Both the lists must be destroyed by the caller. The 
    op_status list cannot be reused with another api call. It must be created 
    and destroyed for each api call.

    One or more attributes of an already registered resource can be modified by
    passing the CLSCRS_FLAG_REG_UPDATE flag. The flags apply to all resources 
    in the input reslist.
    
  PARAMETERS

    in_reslist [IN]  -  list of resources to be registered
    flags      [IN]  -  CLSCRS_FLAG_REG_UPDATE,
                        CLSCRS_FLAG_QUEUE,
                        CLSCRS_FLAG_FORCE,
                        CLSCRS_FLAG_REG_PURGE,
                        or 0
                        See CLSCRS_FLAG enum for more details.
    msgf       [IN]  -  user message callback, may be null.
    msgarg     [IN]  -  user callback argument, may be null.
    op_status  [OUT] -  resource list holding the status of the register
                        operation for each resource.

  RETURNS
  
    CLSCRS_STAT_SUCCESS      - if all input resources are successfully registered
    CLSCRS_STAT_FAILURE      - if atleast one resource could not be registered
    CLSCRS_STAT_CONNECTION   - if there is a communication error with the crsd 
    CLSCRS_STAT_INVALID_ARGS - if any of the resources in the reslist does not
                               have attributes 

  SINCE
    CRS - 11.2
*/

CLSCRS_STAT
clscrs_register_resource2(clscrs_reslist *in_reslist, uword flags, 
                          clscrs_msgf2 msgf, void *msgarg,
                          clscrs_reslist *op_status);

/* ------------------------ clscrs_unregister_resource --------------------- */

/*
  NAME
    clscrs_unregister_resource   DEPRECATED

  DESCRIPTION
    Unregister the resources in the input list of resource names. The output 
    op_status list contains the results of the unregister operation for each 
    resource.
    
    The caller must create and populate the rqlist and must create the 
    op_status list. Both the lists must be destroyed by the caller. The 
    op_status list cannot be reused with another api call. It must be created 
    and destroyed for each api call.

  PARAMETERS
    rqlist     [IN]  -  list of resources names to be unregistered
    flags      [IN]  -  option flags
    op_status  [OUT] -  resource list holding the status of the unregister
                        operation for each resource.

  RETURNS
    CLSCRS_STAT_SUCCESS    - if all input resources are successfully unregistered
    CLSCRS_STAT_FAILURE    - if atleast one resource could not be unregistered
    CLSCRS_STAT_CONNECTION - if there is a communication error with the crsd 
*/

CLSCRS_STAT 
clscrs_unregister_resource(clscrs_splist *rqlist, uword flags, 
                           clscrs_reslist *op_status);

/* ------------------------ clscrs_unregister_resource2 -------------------- */
/*
  NAME
    clscrs_unregister_resource2

  DESCRIPTION
    Unregister the resources in the input list of resource names. The output 
    op_status list contains the results of the unregister operation for each 
    resource.
    
    The caller must create and populate the rqlist and must create the 
    op_status list. Both the lists must be destroyed by the caller. The 
    op_status list cannot be reused with another api call. It must be created 
    and destroyed for each api call.

  PARAMETERS
    rqlist     [IN]  -  list of resources names to be unregistered
    flags      [IN]  -  option flags
    msgf       [IN]  -  user message callback, may be null.
    msgarg     [IN]  -  user callback argument, may be null.
    op_status  [OUT] -  resource list holding the status of the unregister
                        operation for each resource.

  RETURNS
    CLSCRS_STAT_SUCCESS    - if all input resources are successfully unregistered
    CLSCRS_STAT_FAILURE    - if atleast one resource could not be unregistered
    CLSCRS_STAT_CONNECTION - if there is a communication error with the crsd 

  SINCE
    CRS - 11.2
*/

CLSCRS_STAT 
clscrs_unregister_resource2(clscrs_splist *reslist, uword flags, 
                            clscrs_msgf2 msgf, void *msgarg,
                            clscrs_reslist *op_status);

/* ------------------------------- clscrs_stat ---------------------------- */
/*
  NAME
    clscrs_stat DEPRECATED

  DESCRIPTION
    Get information about the resources identified in rqlist. The resources can
    be restricted to a specific node given in the nodename parameter. The 
    information is returned in the output out_reslist.

    If the rqlist is NULL, information about all registered resources is
    returned.

    If the nodename is not NULL, the result is restricted to resources running
    on the specific node. Otherwise, all nodes in the cluster are considered.

    The res_flags are used to scope the resource domain that is matched. There
    are currently no values for this parameter. In the future, it may ne used
    to get all resources in a dependency tree or resources that meet some
    attribute predicates - e.g.,resources that are disabled, running etc.

    The res_attrs identifies the attributes to be returned in the 
    out_reslist. The caller may create a specific list of attributes of 
    interest or use a pre-defined list returned by one of the macros 
    CLSCRS_ATTRS_NONE, CLSCRS_ATTRS_ALL, CLSCRS_ATTRS_STATE, 
    CLSCRS_ATTRS_SHORT. The attribute list applies to all resources
    identified by the rqlist.

    Any errors for a resource are returned in the output out_reslist.

  PARAMETERS
    rqlist       [IN]    list of resource names to query
    nodename     [IN]    query resources running only on the given node 
                         (optional, may be NULL)
    res_flags    [IN]    flags to restrict the resources being matched
    res_attrs    [IN]    list to specify the attributes to be returned for each
                         resource
    out_reslist  [OUT]   list holding the returned resource information

  RETURNS
    CLSCRS_STAT_SUCCESS     - if information is got for all input resources
    CLSCRS_STAT_FAILURE     - if no information is available for atleast one 
                              resource
    CLSCRS_STAT_CONNECTION  - if there is an error communicating with the crsd 
*/

CLSCRS_STAT
clscrs_stat(clscrs_splist *rqlist, const oratext *nodename, uword res_flags,
            clscrs_splist *res_attrs, clscrs_reslist *out_reslist);

/* ------------------------------- clscrs_stat2 ---------------------------- */
/*
  NAME
    clscrs_stat2

  DESCRIPTION
    Get information about the entities identified in rqlist. The information 
    is returned in the output out_entitylist. If the rqlist is NULL, 
    information about all registered entities is returned.

    The argument 'attrs' is used to list the attributes to be returned in the 
    out_entitylist. The caller may create a specific list of attributes of 
    interest. The attribute list applies to all entities in rqlist. 
    If the attribute list is null, all attributes will be returned.

    The argument 'filter' is either a simple filter or a expression filter.
    A simple filter has two operands and a comparison operator given by the
    enum clscrs_comparator. The operands can be any attr=value pair for 
    the entity. Eg, ENABLED=1 for a server entity.
    A expression filter has two simple filters and an expression operator
    given by the enum clscrs_operator. Eg, CARDINALITY=3 && TYPE=db 
    Note that use of the 'filter' parameter is exclusive with 'qlist': at most
    one of the two can be used to limit the result set at a time.

    The out_entitylist must be created and passed as an empty list.
    Any errors for an entity are returned in the out_entitylist.

  PARAMETERS
    qlist           [IN]    list of entity names to query
    entity          [IN]    the entity queried, specified by clscrs_entity enum
    filter          [IN]    filter to restrict the entities matched.
                            eg. CARDINALITY=3 && TYPE=db
    nodename        [IN]    query resources running on the given node
                            (may be NULL)          
    attrs           [IN]    list of attributes to be returned for
                            each entity (NULL returns all attributes)
    flags           [IN]    flags 
    out_entitylist  [OUT]   list holding the returned entity information

  FLAGS
    CLSCRS_STATFLAG_RESINST Query the resource instances of a resource.
                            This flag is allowed only for resource entity
                            types(clscrs_entity_res). Also, a filter cannot
                            be specified when this flag is passed.

  RETURNS
  
    CLSCRS_STAT_SUCCESS     - if information is got for all input resources
    CLSCRS_STAT_FAILURE     - if no information is available for atleast one 
                              resource
    CLSCRS_STAT_CONNECTION  - if there is an error communicating with the crsd 

  SINCE
    CRS - 11.2

  VERSION COMMENTS
       Ver            Comments
    ------------------------------------------   
      12.1            Supports flags:
                   CLSCRS_STATFLAG_SERVERBYCATEGORY
                   CLSCRS_STATFLAG_CATEGORYBYSERVER

      11.2         Supports all the other flags except 
                   the ones mentioned above.
*/
CLSCRS_STAT
clscrs_stat2(clscrs_splist *qlist, clscrs_entity entity,
             clscrs_filter *filter, const oratext *nodename, 
             clscrs_splist *attrs, uword flags, 
             clscrs_reslist *out_entitylist);

/* ---------------------- clscrs_start_resource ---------------------------- */
/*
  NAME
    clscrs_start_resource  DEPRECATED

  DESCRIPTION
    Tell the CRS system to start a named set of resources.  If node is
    specified, start there, otherwise according to the placement policy for
    the resources in question. If the flag is async, msgf will never be
    called, and OK status will be returned after initiating the call to the
    CRS system, and the actual starts will be asynchronously executed, and
    msgf will never be called.  If flag is not async, and msgf is not null,
    msgf will be driven line at a time with collected output from the start
    programs.  An optional EVENT flag maybe passed in to indicate that this
    this not a request to perform an action, but rather a notification that the
    action has already started. The flag should only be used for a narrow set of
    co-managed resources. 

    
  PARAMETERS
    ctx        [IN]   clscrs context
    names      [IN]   name of resource(s) to start
    num_names  [IN]   number of resources in names[] above
    node       [IN]   the nodename on which to start the resource, may be null
    env        [IN]   environment arguments to the start operation
    msgf       [IN]   user message callback, may be null.
    msgarg     [IN]   user callback argument, may be null.
    flag       [IN]   async, force, event options.

  RETURNS
    status enum
*/

CLSCRS_STAT 
clscrs_start_resource(clscrs_ctx *ctx, 
                      const oratext *name[], sword num_names,
                      const oratext *node, clscrs_env env,
                      clscrs_msgf msgf, void *msgarg, uword flag);


/* ----------------------- clscrs_stop_resource ---------------------------- */
/*
  NAME
    clscrs_stop_resource  DEPRECATED

  DESCRIPTION
    Tell the CRS system to stop a named set of resources.  The flag allows
    async stop. 
    
  PARAMETERS
    ctx        [IN]   clscrs context
    names      [IN]   name of resource(s) to stop
    num_names  [IN]   number of resources in names[] above
    env        [IN]   environment arguments to the stop operation
    msgf       [IN]   user message callback, may be null.
    msgarg     [IN]   user callback argument, may be null.
    flag       [IN]   async, force, event options.

  RETURNS
    status enum
*/

CLSCRS_STAT 
clscrs_stop_resource(clscrs_ctx *ctx,
                     const oratext *names[], sword num_names,
                     clscrs_env env,                 
                     clscrs_msgf msgf,
                     void *msgarg,
                     uword flag);



/* ------------------------ clscrs_fail_resource --------------------------- */
/*
  NAME
    clscrs_fail_resource   DEPRECATED

  DESCRIPTION
    Tells CRS to act as if the identified resources in rqlist have
    failed, without running their check actions. The failure is always
    asynchronous with respect to the caller as are the recovery
    actions performed by the CRSD.  The expanded resources are
    identified and sent back to the caller in the op_status list,
    containing the status of the lookup and permission check of each
    matching resource.  A resource that is explicitly named with no
    pattern matching will have a NOTFOUND error.  A pattern that has
    no matches will return no error.  There will be no valid
    attributes returned for the resources.  Errors will also be
    returned if the caller does not have permission to operate on the
    matching resources. Returns SUCCESS if the operation status of all
    identified resources is SUCCESS, otherwise FAILURE.

  PARAMETERS
    ctx         [IN]     clscrs context
    rqlist      [IN]     list of the resources to fail.
    flags       [IN]     none.
    op_status   [OUT]    resource list structure holding the status of the 
                         fail operation for each resource.

  RETURNS
    status enum.
*/

CLSCRS_STAT 
clscrs_fail_resource( clscrs_splist *in_splist,
                      uword flags,
                      clscrs_reslist *op_status);


/* ----------------------- clscrs_check_resource --------------------------- */
/*
  NAME
    clscrs_check_resource   DEPRECATED

  DESCRIPTION
    THIS IS NOW DEPRECATED. Please use clscrs_check_resource2.
    Tell CRS to run the check actions for the identified resources. A
    delay can be specified in the indicating a time to wait in seconds
    before the check action is fired in the server, after the call
    returns. This can be used to let something run before doing the
    check.  For each resource identified by the in_splist, a status will
    be returned in the op_status, indicating whether the resource
    exists and whether the caller has permission to invoke operations
    on the resource.  A resource that is explicitly named with no
    pattern matching will have a NOTFOUND error.  A pattern that has
    no matches will return no error. There will be no valid attributes
    returned for the resources.  There is no way of knowing exactly
    when the checks will actually be run or be completed.  Returns
    SUCCESS if the operation status of all identified resources is
    SUCCESS, otherwise FAILURE and specific status will be found in
    the op_status entries.  The caller must create the op_status list
    before the call, and destroy it when done.

  PARAMETERS
    in_splist     [IN]      list of resources to check
    delay_secs    [IN]      time to wait before the check is run, as 
                            appropriate to the resource
    flags         [IN]      none.
    op_status     [OUT]     resource list structure holding the status of the 
                            fail operation for each resource.
    
  RETURNS
    status enum.  
*/

CLSCRS_STAT 
clscrs_check_resource( clscrs_splist *in_splist,
                       ub4 delay_seconds,
                       uword flags,
                       clscrs_reslist *op_status);


/* ----------------------- clscrs_check_resource2 -------------------------- */
/*
  NAME
    clscrs_check_resource2 

  DESCRIPTION
    Tell CRS to run the check actions for the identified resources. A
    delay can be specified in the call indicating a time to wait in seconds
    before the check action is fired in the server, after the call
    returns. This can be used to let something run before doing the
    check.  For each resource identified by the rqlist, a status will
    be returned in the op_status, indicating whether the resource
    exists and whether the caller has permission to invoke operations
    on the resource.  A resource that is explicitly named with no
    pattern matching will have a NOTFOUND error.  A pattern that has
    no matches will return no error. There will be no valid attributes
    returned for the resources.  There is no way of knowing exactly
    when the checks will actually be run or be completed.  Returns
    SUCCESS if the operation status of all identified resources is
    SUCCESS, otherwise FAILURE and specific status will be found in
    the op_status entries.  The caller must create the op_status list
    before the call, and destroy it when done.
    Note that if node_name is not valid, a member down error msg would
    be populated in the op_status.
    If the rqlist has a local resource or instance id, and the node_name 
    specified is not the one where the local resource was created, 
    then a res_not_found error msg would be found in the op_status entries.

    For resources:

    1.If node_name IS NOT supplied, then
      For all RIs, issue check on the last server if it exists,
      else, issue it on the node from which the call was made.
    2.If node_name IS supplied, then
      1.If the resource IS NOT relocatable, then send check to
        only those RIs where last server is same as the node_name
        supplied by the user.
      2.If the resource IS relocatable, then send check to all RIs
        where last server is same as the node_name supplied by the user.
        2a.If there are no RIs for case 2, pick a RI that
           is offline, and issue a check for that RI.
        2b.If there are no RIs for case 2a. as well, then
           an error msg would be issued by CRSD. Note that 
           OR_RES_PLACEMENT error would be returned to the API 
           by CRSD in this case.

    For resource instances:

    1.If node_name IS supplied, then
      1a.If RI is online/quandary/unknown, then
         send check only if  the last server is same as
         the node_name supplied by the user, else
         an error msg would be issued by CRSD. Note that 
         OR_RES_PLACEMENT error would be returned to the API 
         by CRSD in this case.
      1b.If resource for that RI IS relocatable, then
         send check for that RI, else
         If resource for that RI IS NOT relocatable,
         then send a check only if last server is same as node_name
         supplied by the user, else an error msg would be issued by CRSD. 
         Note that OR_RES_PLACEMENT error would be returned to the API 
         by CRSD in this case.
    2.If node_name IS NOT supplied, then
      2a.If RI is online/quandary/unknown, send check to
         the last server
      2b.Else, send check to the node where the call was made from.
                                                                                                             
    The call also takes filters instead of a list of resources/RIs.
                                                                      
    Parameters: 
    rqlist        [IN]      list of resources/resource instances to check, 
                            or a filter
    node_name     [IN]      node name where the check needs to be done
    delay_secs    [IN]      time to wait before the check is run, as 
                            appropriate to the resource
    msgf          [IN]      user message callback, may be null.
    flags         [IN]      none.
    op_status     [OUT]     resource list structure holding the status of the 
                            fail operation for each resource.
    
  RETURNS
    CLSCRS_STAT_MISSING_RESNAME - if rqlist is NULL
    CLSCRS_STAT_INVALID_RESNAME - if no entities are specified in rqlist
    CLSCRS_STAT_SUCCESS         - on success
    CLSCRS_STAT_FAILURE         - on any failure

  SINCE
    CRS - 11.2
*/

CLSCRS_STAT 
clscrs_check_resource2( clscrs_splist *rqlist,
                        const oratext *node_name,
                        ub4 delay_seconds,
                        clscrs_msgf2 msgf,
                        uword flags,
                        clscrs_reslist *op_status);


/*************************** stringpair functions **************************/
/* The stringpair and list manipulation functions are not thread safe. Thus,
   the structures cannot be shared across threads */

/*
  NAME
    clscrs_sp_set

  DESCRIPTION
    Change the value part of a stringpair. The new value may be NULL. After the
    call returns, the memory of the value can be recycled.

  PARAMETERS
    sp    [IN]  - stringpair to set the value for
    value [IN]  - value component of the stringpair (may be NULL)

  RETURNS
    clscrsretSUCC   - on success  
    clscrsretBADARG - if the sp argument is null
*/

clscrsret clscrs_sp_set(clscrs_sp *sp, const oratext *value);

/*
  NAME
    clscrs_sp_get

  DESCRIPTION
    Get the name and value components of a stringpair.

  PARAMETERS
    sp    [IN]   - stringpair for which the name and value are being get
    name  [OUT]  - name component of the stringpair
    value [OUT]  - value component of the stringpair (may be NULL)

  RETURNS
    clscrsretSUCC    - on success
    clscrsretNOMEM   - if no memory can be allocated
    clscrsretBADARG  - if the sp argument is null
*/

clscrsret clscrs_sp_get(clscrs_sp *sp, oratext **name, oratext **value);

/*
  NAME
    clscrs_sp_get_value

  DESCRIPTION
    Get the value component of a stringpair.

  PARAMETERS
    sp    [IN]   - stringpair for which the name and value are being get
    value [OUT]  - value component of the stringpair (may be NULL)

  RETURNS
    clscrsretSUCC   - on success
    clscrsretBADARG - if the sp argument is null
*/

clscrsret clscrs_sp_get_value(clscrs_sp *sp, oratext **value);


/***************************** splist functions ***************************/

/*
  NAME
    clscrs_splist_create

  DESCRIPTION
    Create a new stringpair list. The memory for the splist is allocated by 
    the function.

  PARAMETERS
    ctx     [IN]  - clscrs context
    splist  [OUT] - the new stringpair list created

  RETURNS
    clscrsretSUCC    - on success
    clscrsretNOMEM   - if no memory can be allocated
    clscrsretBADCTX  - if the context is null      
*/

clscrsret clscrs_splist_create(clscrs_ctx *ctx, clscrs_splist **splist);

/*
  NAME
    clscrs_splist_create_and_set

  DESCRIPTION
    Create a new stringpair list (splist) and set the name and value for the 
    first stringpair in the list. The memory for the splist is allocated by 
    the function.

  PARAMETERS
    ctx   [IN]  - clscrs context
    name  [IN]  - name component of the stringpair
    value [IN]  - value component of the stringpair (may be NULL)
    sp    [OUT] - the new stringpair created

  RETURNS
    clscrsretSUCC   - on success
    clscrsretNOMEM  - if no memory can be allocated
    clscrsretBADCTX - if the context is null
    clscrsretBADARG - if the name argument is null
*/

clscrsret clscrs_splist_create_and_set(clscrs_ctx *ctx, const oratext *name, 
                                       const oratext *value, 
                                       clscrs_splist **splist);

/*
  NAME
    clscrs_splist_append

  DESCRIPTION
    Add a new stringpair (sp) to a stringpair list (splist). 

  PARAMETERS
    splist  [IN]  - splist to add the new {name, value} stringpair.
    name    [IN]  - name component of the stringpair
    value   [IN]  - value component of the stringpair (may be NULL)

  RETURNS
    clscrsretSUCC    - on success
    clscrsretNOMEM   - if no memory can be allocated
    clscrsretBADARG  - if the name argument is null
*/

clscrsret clscrs_splist_append(clscrs_splist *splist, const oratext *name, 
                               const oratext *value);

/*
  NAME
    clscrs_splist_first

  DESCRIPTION
    Get the first stringpair (sp) from a stringpair list (splist).

  PARAMETERS
    splist  [IN]  - splist to get the first sp from.
    sp      [OUT] - the first sp from the given splist.

  RETURNS
    clscrsretSUCC   - on success
    clscrsretBADARG - if the splist is null
    clscrsretEMPTY  - if there are no stringpair elements in the splist
*/

clscrsret clscrs_splist_first(clscrs_splist *splist, clscrs_sp **sp);

/*
  NAME
    clscrs_splist_next

  DESCRIPTION
    Get the current next stringpair (sp) from a stringpair list (splist). This
    function is called to iterate over the stringpairs in a splist.

  PARAMETERS
    splist  [IN]  - splist to get the next sp from.
    sp      [OUT] - the next sp {name, value} in the given splist.

  RETURNS
    clscrsretSUCC    - on success
    clscrsretBADARG  - if the splist is null
    clscrsretENDLIST - if there are no more stringpair elements in the splist
*/

clscrsret clscrs_splist_next(clscrs_splist *splist, clscrs_sp **sp);

/*
  NAME
    clscrs_splist_replace

  DESCRIPTION
    Replace the value for a stringpair (sp) in a stringpair list (splist).

  PARAMETERS
    splist  [IN]  - splist to add the new {name, value} stringpair.
    name    [IN]  - name for which the value is being replaced.
    value   [IN]  - replacement value for the name (may be NULL)

  RETURNS
    clscrsretSUCC    - on success
    clscrsretNONAME  - if there is no stringpair matching the given name
    clscrsretBADARG  - if the name argument is null
*/

clscrsret clscrs_splist_replace(clscrs_splist *splist, const oratext *name, 
                                  const oratext *value);

/*
  NAME
    clscrs_splist_delete_sp

  DESCRIPTION
    Delete a stringpair (sp) from a stringpair list (splist).

  PARAMETERS
    splist  [IN]  - splist from which to delete the {name, value} stringpair.
    name    [IN]  - name to be deleted from the given splist.

  RETURNS
    clscrsretSUCC   - on success
    clscrsretNONAME - if there is no stringpair matching the given name
    clscrsretBADARG - if the name argument is null
*/

clscrsret clscrs_splist_delete_sp(clscrs_splist *splist, const oratext *name);

/*
  NAME
    clscrs_splist_find

  DESCRIPTION
    Find the value for a stringpair (sp) in a stringpair list (splist).

  PARAMETERS
    splist  [IN]  - splist to look into 
    name    [IN]  - name for which the value is looked up.
    value   [OUT] - value associated with the given name in the given splist

  RETURNS
    clscrsretSUCC   - on success 
    clscrsretNONAME - if there is no stringpair matching the given name
    clscrsretBADARG - if the name argument is null
*/

clscrsret clscrs_splist_find(clscrs_splist *splist, const oratext *name, 
                             oratext **value);

/*
  NAME
    clscrs_splist_count

  DESCRIPTION
    Count the number of stringpairs (sp) in a stringpair list (splist).

  PARAMETERS
    splist  [IN]  - splist to count the number of stringpairs.
    count   [OUT] - the number of stringpairs in the given splist.

  RETURNS
    clscrsretSUCC   - on success  
    clscrsretBADARG - if the splist is null
*/

clscrsret clscrs_splist_count(clscrs_splist *splist, ub4 *count);

/*
  NAME
    clscrs_splist_destroy

  DESCRIPTION
    Free the memory for a stringpair list (splist).

  PARAMETERS
    splist  [IN]  - splist to destroy

  RETURNS
    clscrsretSUCC - on success
*/

clscrsret clscrs_splist_destroy(clscrs_splist **splist);



/************** action functions for what-if API ***************************/

/*
  NAME
    clscrs_action_gettype

  DESCRIPTION
    Determines the type of the action provided. 

  PARAMETERS
    action  [IN]  - the action whose type needs to be determined
    atype   [OUT] - struct which will be filled with result, allocated by caller.

  RETURNS
    clscrsretSUCC    - on success
    clscrsretFAIL    - if the action is invalid or type could not be determined
*/

clscrsret clscrs_action_gettype(clscrs_action *action, clscrs_actiontype *atype); 


/*
  NAME
    clscrs_action_getparams

  DESCRIPTION
    Determines the params of the action provided. The memory for the splist is 
    owned by the function, and it should NOT be deallocated by the caller.

  PARAMETERS
    action  [IN]  - the action whose parameters need to be determined.
    params  [OUT] - splist containing parameters for the action.

  RETURNS
    clscrsretSUCC    - on success
    clscrsretBADARG  - if the action is null
*/

clscrsret clscrs_action_getparams(clscrs_action *action, clscrs_splist **params);



/***************************** actionlist functions ***************************/

/*
  NAME
    clscrs_actionlist_create

  DESCRIPTION
    Creates a new actionlist structure. The memory for the splist is allocated by 
    the function.

  PARAMETERS
    ctx     [IN]  - clscrs context
    alist   [OUT] - the new actionlist created

  RETURNS
    clscrsretSUCC    - on success
    clscrsretNOMEM   - if no memory can be allocated
    clscrsretBADCTX  - if the context is invalid      
*/

clscrsret clscrs_actionlist_create(clscrs_ctx *ctx, clscrs_actionlist **alist); 


/*
  NAME
    clscrs_actionlist_first

  DESCRIPTION
    Get the first action from an actionlist

  PARAMETERS
    alist   [IN] - the actionlist to be operated on
    action [OUT] - the returned first action
 
  RETURNS
    clscrsretSUCC   - on success
    clscrsretBADARG - if the actionlist is null
    clscrsretEMPTY  - if there are no action elements in the list
*/

clscrsret clscrs_actionlist_first(clscrs_actionlist *alist, clscrs_action **action); 


/*
  NAME
    clscrs_actionlist_next

  DESCRIPTION
    Get the next action from an actionlist. This function is used to iterate over
    the actions in an actionlist.

  PARAMETERS
    alist   [IN] - the actionlist to be operated on
    action [OUT] - the returned next action
 
  RETURNS
    clscrsretSUCC    - on success
    clscrsretBADARG  - if the actionlist is null
    clscrsretENDLIST - if there are no more action elements in the list
*/

clscrsret clscrs_actionlist_next(clscrs_actionlist *alist, clscrs_action **next_action); 


/*
  NAME
    clscrs_actionlist_count

  DESCRIPTION
    Count the number of actions available in the actionlist

  PARAMETERS
    alist   [IN] - the actionlist to be operated on
    count  [OUT] - the number of actions in the list
 
  RETURNS
    clscrsretSUCC   - on success  
    clscrsretBADARG - if the actionlist is null
*/

clscrsret clscrs_actionlist_count(clscrs_actionlist *alist, ub4 *count); 


/*
  NAME
    clscrs_actionlist_seqid

  DESCRIPTION
    Gets the sequence ID associated with this actionlist. 

  PARAMETERS
    alist   [IN] - the actionlist to be operated on
    seqid  [OUT] - the sequence ID of this actionlist.
 
  RETURNS
    clscrsretSUCC   - on success  
    clscrsretBADARG - if the actionlist is null
*/

clscrsret clscrs_actionlist_seqid(clscrs_actionlist *alist, ub4 *seqid); 


/*
  NAME
    clscrs_actionlist_destroy

  DESCRIPTION
    Free the memory associated with this actionlist, including all
    the actions contained within the list.

  PARAMETERS
    alist   [IN] - the actionlist to be operated on
 
  RETURNS
    clscrsretSUCC   - on success  
    clscrsretBADARG - if the actionlist is null
*/

clscrsret clscrs_actionlist_destroy(clscrs_actionlist *alist);



/**************************** resource functions ***********************/

/*
  NAME
    clscrs_res_create

  DESCRIPTION
    Create a new resource. The memory for the resource structure is allocated 
    by the function. The memory is freed when a resource list (clscrs_reslist) 
    is destroyed through clscrs_reslist_destroy().

  PARAMETERS
    ctx      [IN]  - clscrs context
    resname  [IN]  - name of the resource
    res      [OUT] - the new resource created

  RETURNS
    clscrsretSUCC    - on success
    clscrsretNOMEM   - if no memory can be allocated
    clscrsretBADCTX  - if the context is null
    clscrsretBADARG  - if the resource name is null
*/

clscrsret clscrs_res_create(clscrs_ctx *ctx, const oratext *resname, 
                            clscrs_res **res);

/*
  NAME
    clscrs_res_get_name

  DESCRIPTION
    Get the name of a resource.

  PARAMETERS
    res   [IN]  - resource for which the name is got
    name  [OUT] - name of the resource

  RETURNS
    clscrsretSUCC    - on success
    clscrsretBADARG  - if the resource argument is null
*/

clscrsret clscrs_res_get_name(clscrs_res *res, oratext **name);

/*
  NAME
    clscrs_res_set_attr

  DESCRIPTION
    Set a resource attribute for a resource.

  PARAMETERS
    res      [IN] - resource for which the attribute is set
    attrname [IN] - name of the resource attribute
    value    [IN] - value for the resource attribute (may be NULL)

  RETURNS
    clscrsretSUCC   - on success
    clscrsretBADARG - if the attribute name is null
    clscrsretNOMEM  - if no memory can be allocated
*/

clscrsret clscrs_res_set_attr(clscrs_res *res, const oratext *attrname, 
                              const oratext *value);

/*
  NAME
    clscrs_res_get_attr

  DESCRIPTION
    Get a resource attribute for a resource.

  PARAMETERS
    res      [IN]  - resource for which the attribute is got
    attrname [IN]  - name of the resource attribute
    value    [OUT] - value for the resource attribute

  RETURNS
    clscrsretSUCC   - on success
    clscrsretBADARG - if the attribute name is null
    clscrsretNOMEM  - if no memory can be allocated
*/

clscrsret clscrs_res_get_attr(clscrs_res *res, const oratext *attrname, 
                              oratext **value);

/*
  NAME
    clscrs_res_get_attr_list

  DESCRIPTION
    Get the attribute list for a resource. The attribute list is a list of 
    stringpairs. The client does not allocate the memory for attrlist.

  PARAMETERS
    res      [IN]  - resource for which the attribute list is got
    attrlist [OUT] - list of attributes for the given resource

  RETURNS
    clscrsretSUCC    - on success  
    clscrsretBADARG  - if the resource is null
    clscrsretNOMEM   - if no memory can be allocated
    clscrsretNOATTRS - if there are no attributes set for the resource
*/

clscrsret clscrs_res_get_attr_list(clscrs_res *res, clscrs_splist **attrlist);

/*
  NAME
    clscrs_res_set_attr_list

  DESCRIPTION
    Set the attribute list for a resource. The attribute list is a list of 
    stringpairs. The list is created from the clscrs_splist_create call.

  PARAMETERS
    res      [IN]  - resource for which the attribute list is set
    attrlist [IN]  - list of attributes to be set for the given resource

  RETURNS
    clscrsretSUCC   - on success  
    clscrsretBADARG - if the resource is null
*/

clscrsret clscrs_res_set_attr_list(clscrs_res *res, clscrs_splist *attrlist);

/*
  NAME
    clscrs_res_set_reslist

  DESCRIPTION
    Set the resource list for a resource. 
    The list is created from the clscrs_reslist_create call.

  PARAMETERS
    res     [IN]  - resource for which the resource list is set
    reslist [IN]  - resource list to be set for the given resource

  RETURNS
    clscrsretSUCC   - on success  
    clscrsretBADARG - if the resource is null
*/

clscrsret clscrs_res_set_reslist(clscrs_res *res, clscrs_reslist *reslist);

/*
  NAME
    clscrs_res_get_reslist

  DESCRIPTION
    Get the resource list for a resource. 

  PARAMETERS
    res     [IN]  - resource for which the resource list is set
    reslist [IN]  - resource list to be returned for the given resource

  RETURNS
    clscrsretSUCC   - on success  
    clscrsretBADARG - if the resource is null
*/

clscrsret clscrs_res_get_reslist(clscrs_res *res, clscrs_reslist **reslist);

/*
  NAME
    clscrs_res_attr_count

  DESCRIPTION
    Get the number of attributes for a resource.

  PARAMETERS
    res    [IN]  - resource for which number of attributes is got
    count  [OUT] - number of attributes for the given resource

  RETURNS
    clscrsretSUCC   - on success
    clscrsretBADARG - if the resource is null
*/

clscrsret clscrs_res_attr_count(clscrs_res *res, ub4 *count);

/*
  NAME
    clscrs_res_get_op_status

  DESCRIPTION
    Get the status of an operation for an entity. This function
    was originaly designed to work on op_status for resource
    operations. It has since been extended to cover every supported
    entity kind alike.

  PARAMETERS
    res     [IN]  - the entity for which the operation status is got
    status  [OUT] - status of an operation on the given entity. This 
                    is a programmatic return code, indicating success
                    or a high-level problem kind. Please see 
                    CLSCRS_STAT_* constants above for the possible values.
    msg     [OUT] - DEPRECATED: must not be used. Instead, pass a
                    clscrs_msgf2 callback into the API call and that will 
                    receive human-readable messages. 
                    Used to be: text message for the result of an 
                    operation on the resource

           
  RETURNS
    clscrsretSUCC   - on success
    clscrsretNOMEM  - if no memory can be allocated
    clscrsretNOMSG  - if there is no msg available
    clscrsretBADARG - if the resource is null
*/

clscrsret clscrs_res_get_op_status(clscrs_res *res, CLSCRS_STAT *status, 
                                   oratext **msg);

/*
  NAME
    clscrs_res_get_registered

  DESCRIPTION
    Get the registration status of a resource.

  PARAMETERS
    res         [IN]  - resource for which the operation status is set
    registered  [OUT] - boolean indicating whether the resource is registered

  RETURNS
    clscrsretSUCC   - on success
    clscrsretNOMEM  - if no memory can be allocated
    clscrsretBADARG - if the resource is null
*/

clscrsret clscrs_res_get_registered(clscrs_res *res, boolean *registered);

/*
  NAME
    clscrs_res_get_node_list

  DESCRIPTION
    Get the nodelist currently hosting the resource, or NULL if there is no 
    host for the resource or there are no attributes. The caller need not 
    allocate memory for the nodelist.

  PARAMETERS
    res      [IN]  - resource for which the nodelist is got
    nodelist [OUT] - splist holding the node(s)

  RETURNS
    clscrsretSUCC   - on success
    clscrsretBADARG - if the resource is null
*/

clscrsret clscrs_res_get_node_list(clscrs_res *res, clscrs_splist **nodelist);

/*
  NAME
    clscrs_res_destroy

  DESCRIPTION
    Free the memory for a resource.

  PARAMETERS
    res      [IN] - resource for which the memory is freed

  RETURNS
    clscrsretSUCC - on success
*/

clscrsret clscrs_res_destroy(clscrs_res **res);


/************************* reslist functions ******************************/

/*
  NAME
    clscrs_reslist_create

  DESCRIPTION
    Create a new resource list. The memory for the resource list is allocated 
    by the function.

  PARAMETERS
    ctx     [IN]  - clscrs context
    reslist [OUT] - resource list (empty) that is created

  RETURNS
    clscrsretSUCC   - on success
    clscrsretNOMEM  - if no memory can be allocated
    clscrsretBADCTX - if the context is null
*/

clscrsret clscrs_reslist_create(clscrs_ctx *ctx, clscrs_reslist **reslist);

/*
  NAME
    clscrs_reslist_isempty

  DESCRIPTION
    Check if resource list is empty.

  PARAMETERS

    reslist [in] - resource list to add the resource to

  RETURNS

    clscrsretSUCC               if reslist is empty 
    clscrsretBADARG             if reslist is null
    clscrsretFAIL               if reslist is not empty
*/

clscrsret clscrs_reslist_isempty(clscrs_reslist *reslist);

/*
  NAME
    clscrs_reslist_append

  DESCRIPTION
    Add a resource to a resource list.

  PARAMETERS
    reslist [IN] - resource list to add the resource to
    res     [IN] - resource to add

  RETURNS
    clscrsretSUCC      - on success
    clscrsretBADARG    - if reslist is null
    clscrsretRESEXISTS - if the resource already exists in reslist
*/

clscrsret clscrs_reslist_append(clscrs_reslist *reslist, clscrs_res *res);

/*
  NAME
    clscrs_reslist_first

  DESCRIPTION
    Get the first resource on a resource list. 

  PARAMETERS
    reslist [IN]  - resource list for which the first resource is got
    res     [OUT] - the first resouce on the resource list

  RETURNS
    clscrsretSUCC   - on success.
    clscrsretBADARG - if reslist is null
    clscrsretEMPTY  - if there are no resources in the list
*/

clscrsret clscrs_reslist_first(clscrs_reslist *reslist, clscrs_res **res);

/*
  NAME
    clscrs_reslist_next

  DESCRIPTION
    Get the current next resource from the resource list. This function is 
    called to iterate over the resources in a resource list.

  PARAMETERS
    reslist [IN]  - resource list for which the first resource is got
    res     [OUT] - the next resouce on the resource list

  RETURNS
    clscrsretSUCC    - on success
    clscrsretBADARG  - if reslist is null
    clscrsretENDLIST - if there are no more resources in the list
*/

clscrsret clscrs_reslist_next(clscrs_reslist *reslist, clscrs_res **res);

/*
  NAME
    clscrs_reslist_find

  DESCRIPTION
    Find a resource in a resource list.

  PARAMETERS
    reslist [IN]  - resource list (empty) that is created
    name    [IN]  - name of the resource that is being get
    res     [OUT] - the resouce corresponding to the given name

  RETURNS
    clscrsretSUCC   - on success
    clscrsretNORES  - if the resource is not found
    clscrsretBADARG - if reslist or name is null
*/

clscrsret clscrs_reslist_find(clscrs_reslist *reslist, const oratext *name, 
                                clscrs_res **res);
/*
  NAME
    clscrs_reslist_count

  DESCRIPTION
    Count the number of resources in a resource list.

  PARAMETERS
    reslist [IN]  - resource list for which the count is got
    count   [OUT] - number of resources in the resource list

  RETURNS
    clscrsretSUCC   - on success
    clscrsretBADARG - if reslist is null
*/

clscrsret clscrs_reslist_count(clscrs_reslist *reslist, ub4 *count);

/*
  NAME
    clscrs_reslist_delete_res

  DESCRIPTION
    Delete a resource from a resource list.

  PARAMETERS
    reslist [IN] - resource list from which to delete the resource
    name    [IN] - name of the resource to delete 

  RETURNS
    clscrsretSUCC   - on success
    clscrsretBADARG - if reslist or name is null
    clscrsretNORES  - if the resource is not in reslist
*/

clscrsret clscrs_reslist_delete_res(clscrs_reslist *reslist, 
                                    const oratext *name);

/*
  NAME
    clscrs_reslist_destroy

  DESCRIPTION
    Free the memory for a resource list.

  PARAMETERS
    reslist [IN]  - resource list for which the memory is to be freed

  RETURNS
    clscrsretSUCC - on success
*/

clscrsret clscrs_reslist_destroy(clscrs_reslist **reslist);

/*
  NAME
    clscrs_reslist_get_seqnumber

  DESCRIPTION
    Returns the event sequence number from a resource list.
    This sequence number is used by events and queries to
    provide a way to synchronize results of a query with its events.
    The resource list that is passed in to this function MUST be the
    one that has been returned from a stat2 call. Also, the sequence 
    number would be present in the resource list only if the stat2 call
    has succeded.

  PARAMETERS
    reslist  [IN]  - resource list that has the sequence number
    seqnum   [OUT] - sequence number

  RETURNS
    clscrsretSUCC   - on success
    clscrsretBADARG - if reslist or seqnum is null
*/

clscrsret clscrs_reslist_get_seqnumber(clscrs_reslist *reslist, 
                                       sb4 *seqnum);

/*
  NAME
    clscrs_get_error_message DEPRECATED 

  DESCRIPTION
    Retrieve the error message corresponding to the return codes from a clscrs 
    api. The memory for the error message is allocated by the caller. If the 
    buffer is not large enough, the length is returned in msg_len.
    
  PARAMETERS
    ctx      [IN]    - clscrs context
    err_code [IN]    - error code returned from the clscrs api
    msg      [OUT]   - message corresponding to err_code
    msg_len  [INOUT] - length of the message buffer.

  RETURNS
    clscrsretSUCC - on success
*/

clscrsret clscrs_get_error_message(clscrs_ctx *ctx, clscrsret err_code, 
                                   oratext *msg, sb4 msg_len);

/**************************** resource type functions ***********************/

/*
  NAME
    clscrs_type_create

  DESCRIPTION
    Creates a new resource type. The memory for the resource structure is 
    allocated by the function. The memory is freed when a resource list 
    (clscrs_reslist) is destroyed through clscrs_reslist_destroy().

  PARAMETERS
    ctx       [IN]  - clscrs context
    type_name [IN]  - name of the resource type
    res       [OUT] - the new resource type created

  RETURNS
    clscrsretSUCC   - on success
    clscrsretNOMEM  - if no memory can be allocated
    clscrsretBADCTX - if the context is null
    clscrsretBADARG - if the resource name is null
*/
clscrsret clscrs_type_create(clscrs_ctx *ctx, const oratext *type_name, 
                             clscrs_res **type);


/*
  NAME
    clscrs_type_set_attr

  DESCRIPTION
    Adds an attribute to a resource type. 

  PARAMETERS
    type       [IN]  - the resource type for which the attribute is set
    attrname   [IN]  - name of an attribute in the resource type
    dtype      [IN]  - the data type of the attribute value (integer, string)
    defaultval [IN]  - the default value for the attribute. If a null value
                       is specified, that attribute value becomes required
                       when a resource of the type is created. For 
                       example, an empty string ("") is a valid default value 
                       for string values.
    flags      [IN] -  flags indicating the attribute properties.
                       At the moment, the following flags can be specified:              
                       - CLSCRS_TYPEFLAG_READONLY
                      
  RETURNS
    clscrsretSUCC   - on success
    clscrsretNOMEM  - if no memory can be allocated
    clscrsretBADCTX - if the context is null
    clscrsretBADARG - if the arguments are bad
*/
clscrsret clscrs_type_set_attr(clscrs_res *type, const oratext *attrname, 
                               clscrs_dtype dtype, const oratext *defaultval,
                               uword flags);

/*
  NAME
    clscrs_type_get_attr

  DESCRIPTION
    Get the value/properties of a resource type attribute

  PARAMETERS
    type       [IN]  - resource type for which the attribute is got
    attrname   [IN]  - name of the resource type attribute
    dtype      [OUT] - the data type of the attribute value (integer, string)
    defaultval [OUT] - the default value for the attribute. If a null value
                       is specified, that attribute value becomes required
                       when a resource of the type is created.
                       For attributes that are not mandatory during resource
                       creation, an empty string ("") must be passed.
                       Memory needs to be allocated by the caller.
    flags      [OUT] - flags indicating the attribute properties like readonly,etc.

  RETURNS
    clscrsretSUCC   - on success
    clscrsretNOMEM  - if no memory can be allocated
    clscrsretBADCTX - if the context is null
    clscrsretBADARG - if the arguments are bad
*/
clscrsret clscrs_type_get_attr(clscrs_res *type, const oratext *attrname, 
                               clscrs_dtype *dtype, oratext *defaultval,
                               uword *flags);

/*
  NAME
    clscrs_register_type

  DESCRIPTION
    Register the resource types in the input resource list. The attributes for
    the resource types are encapsulated in the input resource list. The output
    op_status list contains the results of the register operation for each 
    resource type, and contains no valid resource attributes. 

    The caller must create and populate the typelist and must create the 
    op_status list. Both the lists must be destroyed by the caller. The 
    op_status list cannot be reused with another api call. It must be created 
    and destroyed for each api call.

    One or more attributes of an already registered resource type can be 
    modified by passing the CLSCRS_FLAG_REG_UPDATE flag. The flags apply to 
    all resources in the input typelist.
    
  PARAMETERS
    typelist   [IN]  -  list of resource types to be registered
    flags      [IN]  -  CLSCRS_FLAG_REG_UPDATE,
                        CLSCRS_FLAG_QUEUE,
                        CLSCRS_FLAG_FORCE,
                        or 0
                        See CLSCRS_FLAG enum for more details.
    msgf       [IN]  -  user message callback, may be null.
    msgarg     [IN]  -  user callback argument, may be null.
    op_status  [OUT] -  resource list holding the status of the register
                        operation for each resource type.

  RETURNS
    CLSCRS_STAT_SUCCESS    - if all resource types are successfully registered
    CLSCRS_STAT_FAILURE    - if atleast one resource type could not be registered
    CLSCRS_STAT_CONNECTION - if there is a communication error with the crsd 

  SINCE
    CRS - 11.2
*/

CLSCRS_STAT 
clscrs_register_type(clscrs_reslist *typelist, uword flags,
                     clscrs_msgf2 msgf, void *msgarg,
                     clscrs_reslist *op_status);

/*
  NAME
    clscrs_unregister_type

  DESCRIPTION
    Unregister the resource types in the input list. The output 
    op_status list contains the results of the unregister operation for each 
    resource type.
    
    The caller must create and populate the typelist and must create the 
    op_status list. Both the lists must be destroyed by the caller. The 
    op_status list cannot be reused with another api call. It must be created 
    and destroyed for each api call.

  PARAMETERS
    typelist  [IN]  -  list of resource types to be unregistered
    flags     [IN]  -  option flags
    msgf      [IN]  -  user message callback, may be null.
    msgarg    [IN]  -  user callback argument, may be null.
    op_status [OUT] -  resource list holding the status of the unregister
                        operation for each resource type.

  RETURNS
    CLSCRS_STAT_SUCCESS    - if all resource types are successfully unregistered
    CLSCRS_STAT_FAILURE    - if atleast one resource type could not be unregistered
    CLSCRS_STAT_CONNECTION - if there is a communication error with the crsd 

  SINCE
    CRS - 11.2
*/
CLSCRS_STAT 
clscrs_unregister_type(clscrs_splist *typelist, uword flags,
                       clscrs_msgf2 msgf, void *msgarg,
                       clscrs_reslist *op_status);

/*
  NAME
    clscrs_type_destroy

  DESCRIPTION
    Free the memory for a resource type structure.

  PARAMETERS
    restype      [IN] - resource type for which the memory is freed

  RETURNS
    clscrsretSUCC - on success
*/

clscrsret clscrs_type_destroy(clscrs_res **restype);

/**************************** server pool functions ***********************/
/*
  NAME
    clscrs_serverpool_create

  DESCRIPTION
    Creates a new server pool. The memory for the server pool structure is 
    allocated by the function. The allocated memory is freed via clscrs_res_destroy
    or clscrs_reslist_destroy().

  PARAMETERS
    ctx      [IN]  - clscrs context
    sp_name  [IN]  - name of the resource type
    sp       [OUT] - the new server pool created

  RETURNS
    clscrsretSUCC   - on success
    clscrsretNOMEM  - if no memory can be allocated
    clscrsretBADCTX - if the context is null
    clscrsretBADARG - if the resource name is null
*/
clscrsret clscrs_serverpool_create(clscrs_ctx *ctx, const oratext *sp_name, 
                           clscrs_res **sp);


/*
  NAME
    clscrs_serverpool_destroy

  DESCRIPTION
    Free the memory for a server pool structure.

  PARAMETERS
    sp      [IN] - server pool for which the memory is freed

  RETURNS
    clscrsretSUCC - on success
*/
clscrsret clscrs_serverpool_destroy(clscrs_res **sp);


/*
  NAME
    clscrs_register_serverpool

  DESCRIPTION
    Register a server pool for the input list of servers. The list of servers
    is packed in the serverlist argument.The output op_status struct contains 
    the results of the register operation and contains no valid resource 
    attributes. 

    The caller must create and populate the serverlist and must create the 
    op_status res structure. Both the lists must be destroyed by the caller. 
    The op_status cannot be reused with another api call. It must be created 
    and destroyed for each api call.

    One or more servers of an already registered server pools can be 
    modified by passing the CLSCRS_FLAG_REG_UPDATE flag. 
    
  PARAMETERS
    splist     [IN]  - list of server pools
    flags      [IN]  - CLSCRS_FLAG_REG_UPDATE,
                       CLSCRS_FLAG_QUEUE,
                       CLSCRS_FLAG_FORCE,
                       or 0
                       See CLSCRS_FLAG enum for more details.
    msgf       [IN]  - user message callback, may be null.
    msgarg     [IN]  - user callback argument, may be null.
    op_status  [OUT] - resource structure holding the status of the operation

  RETURNS
    CLSCRS_STAT_SUCCESS    - if the server pool is successfully registered
    CLSCRS_STAT_FAILURE    - if the server pool could not be registered
    CLSCRS_STAT_CONNECTION - if there is a communication error with the crsd 

  SINCE
    CRS - 11.2
*/
CLSCRS_STAT
clscrs_register_serverpool(clscrs_reslist *splist, uword flags,
                           clscrs_msgf2 msgf, void *msgarg,
                           clscrs_reslist *op_status);



/*
  NAME
    clscrs_unregister_serverpool

  DESCRIPTION
    Unregister the given server pool. The caller must create the op_status 
    structure and destroy.

  PARAMETERS
    splist    [IN]  -  name of the server pool to unregister
    flags     [IN]  -  option flags
    msgf      [IN]  -  user message callback, may be null.
    msgarg    [IN]  -  user callback argument, may be null.
    op_status [OUT] -  resource structure holding the status of the unregister
                       operation.

  RETURNS
    CLSCRS_STAT_SUCCESS    - if the server pool is successfully unregistered
    CLSCRS_STAT_FAILURE    - if the server pool could not be unregistered
    CLSCRS_STAT_CONNECTION - if there is a communication error with the crsd 

  SINCE
    CRS - 11.2
*/
CLSCRS_STAT 
clscrs_unregister_serverpool(clscrs_splist *splist, uword flags,
                             clscrs_msgf2 msgf, void *msgarg,
                             clscrs_reslist *op_status);



/*
  NAME
    clscrs_entity_id_create

  DESCRIPTION
    A entity id fully identifies an entity. It contains 
    information that the CRS system can use to fully identify a resource
    that has either cardinality or degree or both greater than one. It also
    identifies other entities like resource types, server pools etc.
    
    The function allocates the memory for id which can be freed by calling
    clscrs_entity_id_destroy().

  PARAMETERS
    ctx          [IN]  - clscrs context 
    name         [IN]  - name of the entity
    entity       [IN]  - the entity for which the id is to be obtained
    id           [OUT] - entity id

  RETURNS
    clscrsretSUCC   - on success
    clscrsretNOMEM  - if no memory can be allocated
    clscrsretBADCTX - if the context is null
    clscrsretBADARG - if the entity name or id is null
*/
clscrsret clscrs_entity_id_create(clscrs_ctx *ctx, const oratext *name,
                                  clscrs_entity entity, oratext **id);

/*
  NAME
    clscrs_entity_id_destroy

  DESCRIPTION
    Frees the memory associated with an entity id created from 
    clscrs_entity_id_create().

  PARAMETERS
    id     [IN]  -  the entity id to free memory

  RETURNS
    clscrsretSUCC   - on success
    clscrsretBADCTX - if the context is null
*/
clscrsret clscrs_entity_id_destroy(clscrs_ctx *ctx, oratext **id);

/***** new functions for the deprecated functions above ***********************/
/*
  NAME
    clscrs_start_resource2

  DESCRIPTION
    Tell the CRS system to start a named set of resources.  If node is
  specified, start there, otherwise according to the placement policy for
  the resources in question. If the flag is async, the user supplied callback
  function msgf will never be called. An OK status will be returned after 
  initiating the call to the CRS system, and the actual starts will be 
  asynchronously executed.

  If flag is not async, and msgf is not null,
  msgf will be driven line at a time with collected output from the start
  programs.  An optional EVENT flag maybe passed in to indicate that this
  this not a request to perform an action, but rather a notification that the
  action has already started. The flag should only be used for a narrow set of
  co-managed resources.

  In the first argument, either a list of resource ids or a filter can be
  given. Giving a filter will allow for the query and action to take place
  in a single call. The filter searches all registered resources.
    
  PARAMETERS
    ridlist    [IN]  -  list of resources or resource instance ids to start,
                        or a filter
    node       [IN]  -  the nodename on which to start the resource, may be null
    env        [IN]  -  environment arguments to the start operation
    msgf       [IN]  -  user message callback, may be null.
    msgarg     [IN]  -  user callback argument, may be null.
    flag       [IN]  -  async, force, event options.
    op_status  [OUT] -  resource list holding the status of the register
                        operation for each resource.

  RETURNS
    CLSCRS_STAT_INVALID_RESNAME       - if ridlist is empty
    CLSCRS_STAT_AUTHORIZATION_FAILURE - on authorization failure
    CLSCRS_STAT_SUCCESS               - if the the request completed 
                                        successfully for all requested 
                                        entities 
                                        or, if no entities matched a filter
    CLSCRS_STAT_FAILURE               - if one or more resources/resource 
                                        instances were not started
                                        successfully
    CLSCRS_STAT_CONNECTION            - if there is a communication error
                                        with the crsd 

  SINCE
    CRS - 11.2

  VERSION COMMENTS
       Ver            Comments
    ------------------------------------------   
      12.1            Supports flags:
                   CLSCRS_T_HA_PREPARE
                   CLSCRS_T_HA_FINISH
                   CLSCRS_NO_STATE_RESTORATION

      11.2         Supports all the other flags except 
                   the ones mentioned above.
*/
CLSCRS_STAT 
clscrs_start_resource2(clscrs_splist *ridlist, const oratext *node, 
                       clscrs_env env, clscrs_msgf2 msgf, void *msgarg, 
                       uword flag, clscrs_reslist *op_status);

/*
  NAME
    clscrs_restart_resource

  DESCRIPTION
    Tell the CRS system to restart a named set of resources. 

  If the flag is async, the user supplied callback
  function msgf will never be called. An OK status will be returned after 
  initiating the call to the CRS system, and the actual restarts will be 
  asynchronously executed.

  If flag is not async, and msgf is not null,
  msgf will be driven line at a time with collected output from the restart
  programs.  An optional EVENT flag maybe passed in to indicate that this
  this not a request to perform an action, but rather a notification that the
  action has already started. The flag should only be used for a narrow set of
  co-managed resources.

  In the first argument, either a list of resource ids or a filter can be
  given. Giving a filter will allow for the query and action to take place
  in a single call. The filter searches all registered resources.
    
  PARAMETERS
    ridlist    [IN]  -  list of resources or resource instance ids to restart,
                        or a filter
    env        [IN]  -  environment arguments to the restart operation
    msgf       [IN]  -  user message callback, may be null.
    msgarg     [IN]  -  user callback argument, may be null.
    flag       [IN]  -  async, force, event options.
    op_status  [OUT] -  resource list holding the status of the register
                        operation for each resource.

  RETURNS
    CLSCRS_STAT_INVALID_RESNAME       - if ridlist is empty
    CLSCRS_STAT_AUTHORIZATION_FAILURE - on authorization failure
    CLSCRS_STAT_SUCCESS               - if the the request completed 
                                        successfully for all requested 
                                        entities 
                                        or, if no entities matched a filter
    CLSCRS_STAT_FAILURE               - if one or more resources/resource 
                                        instances were not restarted
                                        successfully
    CLSCRS_STAT_CONNECTION            - if there is a communication error
                                        with the crsd 
    CLSCRS_STAT_WRONG_ACTIVE_VERSION  - if used before the cluster has been                     
                                        upgraded to 12

  SINCE
    CRS - 12.1

*/

CLSCRS_STAT 
clscrs_restart_resource(clscrs_splist *ridlist, 
                       clscrs_env env, clscrs_msgf2 msgf, void *msgarg, 
                       uword flag, clscrs_reslist *op_status);


/*
  NAME clscrs_start_resource_in_pools

  DESCRIPTION

  Starts the resources in  server pools. Functionality is same as that of 
  "clscrs_start_resource2" except that this function takes in an additional
  argument which is a list of server pool names and does NOT take in "node"
  as an argument.
  

  PARAMETERS
    ridlist    [IN]  -  list of resources or resource instance ids to start,
                        or a filter
    spoollist  [IN]  -  list of server pool names or a filter of server pools 
                        where resources will be started.
    env        [IN]  -  environment arguments to the start operation
    msgf       [IN]  -  user message callback, may be null.
    msgarg     [IN]  -  user callback argument, may be null.
    flag       [IN]  -  async, force, event options.
    op_status  [OUT] -  resource list holding the status of the start
                        operation for each resource.
  RETURNS
    CLSCRS_STAT_INVALID_RESNAME       - if ridlist is empty
    CLSCRS_STAT_AUTHORIZATION_FAILURE - on authorization failure
    CLSCRS_STAT_SUCCESS               - if the the request completed 
                                        successfully for all requested 
                                        entities 
                                        or, if no entities matched a filter
    CLSCRS_STAT_FAILURE               - if one or more resources/resource 
                                        instances were not started
                                        successfully
    CLSCRS_STAT_CONNECTION            - if there is a communication error
                                        with the crsd 
  SINCE
    CRS - 12.1
*/

CLSCRS_STAT 
clscrs_start_resource_in_pools(clscrs_splist *ridlist,clscrs_splist *spoollist,
                       clscrs_env env,clscrs_msgf2 msgf2, void *msgarg,
                       uword flags, clscrs_reslist *op_status);
/*
  NAME
    clscrs_stop_resource2

  DESCRIPTION
  Tell the CRS system to stop a named set of resources.  The flag allows
  async stop.
    
  In the first argument, either a list of resource ids or a filter can be
  given. Giving a filter will allow for the query and action to take place
  in a single call. The filter searches all registered resources.


  PARAMETERS
    ridlist    [IN]  -  list of resources or resource instance ids to stop,
                        or a filter
    env        [IN]  -  environment arguments to the stop operation
    msgf       [IN]  -  user message callback, may be null.
    msgarg     [IN]  -  user callback argument, may be null.
    flag       [IN]  -  async, force, event options.
    op_status  [OUT] -  resource list holding the status of the register
                        operation for each resource.  

  RETURNS
    CLSCRS_STAT_INVALID_RESNAME       - if ridlist is empty
    CLSCRS_STAT_AUTHORIZATION_FAILURE - on authorization failure
    CLSCRS_STAT_SUCCESS               - if the the request completed 
                                        successfully for all requested 
                                        entities 
                                        or, if no entities matched a filter
    CLSCRS_STAT_FAILURE               - if one or more resources/resource 
                                        instances were not stopped
                                        successfully
    CLSCRS_STAT_CONNECTION            - if there is a communication error
                                        with the crsd

  SINCE
    CRS - 11.2

  VERSION COMMENTS
       Ver            Comments
    ------------------------------------------   
      12.1            Supports flags:
                   CLSCRS_T_HA_PREPARE
                   CLSCRS_T_HA_FINISH
                   CLSCRS_NO_STATE_RESTORATION

      11.2         Supports all the other flags except 
                   the ones mentioned above.
*/
CLSCRS_STAT 
clscrs_stop_resource2(clscrs_splist *ridlist, clscrs_env env,
                      clscrs_msgf2 msgf, void *msgarg, uword flag, 
                      clscrs_reslist *op_status);

/*
  NAME

clscrs_stop_resource_in_pools

  DESCRIPTION

  Tell the CRS system to stop a named set of resources in server pools.
  Functionality is pretty much same  as clscrs_stop_resource2() except
  that this function takes an extra argument: spoollist, which is a list
  of server pool names or a filter based on which a list of server
  pools will be generated. This does NOT take 'node' as an argment.
 
  This is available only from CRS 12.1 onwards.
 
  PARAMETERS
    ridlist    [IN]  -  list of resources or resource instance ids to stop,
                        or a filter
    spoollist  [IN]  -  list of server pool names where resource is running to stop,
                        or a filter

    env        [IN]  -  environment arguments to the stop operation
    msgf       [IN]  -  user message callback, may be null.
    msgarg     [IN]  -  user callback argument, may be null.
    flag       [IN]  -  async, force, event options.
    op_status  [OUT] -  resource list holding the status of the stop
                        operation for each resource.

  RETURNS
    CLSCRS_STAT_INVALID_RESNAME       - if ridlist is empty
    CLSCRS_STAT_AUTHORIZATION_FAILURE - on authorization failure
    CLSCRS_STAT_SUCCESS               - if the the request completed
                                        successfully for all requested
                                        entities
                                        or, if no entities matched a filter
    CLSCRS_STAT_FAILURE               - if one or more resources/resource
                                        instances were not stopped
                                        successfully
    CLSCRS_STAT_CONNECTION            - if there is a communication error
                                        with the crsd
  SINCE
    CRS - 12.1
*/

CLSCRS_STAT 
clscrs_stop_resource_in_pools(clscrs_splist *ridlist,clscrs_splist *spoollist,
                      clscrs_env env,clscrs_msgf2 msgf2, void *msgarg,
                      uword flags, clscrs_reslist *op_status);

/*
  NAME
    clscrs_request_action

  DESCRIPTION
  Tell the CRS system to run an action on a named set of resources.
    
  In the first argument, either a list of resource ids or a filter can be
  given. Giving a filter will allow for the query and action to take place
  in a single call. The filter searches all registered resources.

  PARAMETERS
    action_name [IN]  -  name of the action to be performed
    ridlist     [IN]  -  list of resources or resource instance ids to stop,
                         or a filter
    env         [IN]  -  environment arguments to the stop operation
    msgf        [IN]  -  user message callback, may be null.
    msgarg      [IN]  -  user callback argument, may be null.
    flag        [IN]  -  async, queue.
    op_status   [OUT] -  resource list holding the status of the action
                         operation for each resource.  

  RETURNS
    CLSCRS_STAT_INVALID_RESNAME       - if ridlist is empty
    CLSCRS_STAT_AUTHORIZATION_FAILURE - on authorization failure
    CLSCRS_STAT_SUCCESS               - if the the request completed 
                                        successfully for all requested 
                                        entities 
                                        or, if no entities matched a filter
    CLSCRS_STAT_FAILURE               - if one or more resources/resource 
                                        instances were not stopped
                                        successfully
    CLSCRS_STAT_CONNECTION            - if there is a communication error
                                        with the crsd

  SINCE
    CRS - 12.1

*/
CLSCRS_STAT 
clscrs_request_action(oratext *action_name, clscrs_splist *ridlist, 
                      clscrs_env env, clscrs_msgf2 msgf, 
                      void *msgarg, uword flag, clscrs_reslist *op_status);


/*
  NAME
    clscrs_relocate_resource2

  DESCRIPTION
    Relocate the list of resource ids. The resource ids must correspond to
    resource instances or a singleton resource(CARDINALITY=1). The
    behavior for relocating a resource of multiple cardinality is undefined.

    The destination node is the node to which the resource is to be
    relocated. If no node is given, a suitable node is selected by CRS.

    In the first argument, either a list of resource ids or a filter can be
    given. Giving a filter will allow for the query and action to take place
    in a single call. The filter searches all registered resources.

  PARAMETERS
    resid_list [IN]  -  list of resources  or resource instance ids to relocate,
                        or a filter
    destnode   [IN]  -  destination node to relocate to, may be null
    env        [IN]  -  environment arguments to the stop operation
    msgf       [IN]  -  user message callback, may be null.
    msgarg     [IN]  -  user callback argument, may be null.
    flags      [IN]  -  async, queue flags
    op_status  [OUT] -  resource list holding the status of the register
                        operation for each resource.  

  RETURNS
    CLSCRS_STAT_INVALID_RESNAME       - if resid_list is empty
    CLSCRS_STAT_AUTHORIZATION_FAILURE - on authorization failure
    CLSCRS_STAT_SUCCESS               - if the the request completed 
                                        successfully for all requested 
                                        entities 
                                        or, if no entities matched a filter
    CLSCRS_STAT_FAILURE               - if one or more resources/resource 
                                        instances were not relocated
                                        successfully
    CLSCRS_STAT_CONNECTION            - if there is a communication error
                                        with the crsd

  SINCE
    CRS - 11.2
*/
CLSCRS_STAT 
clscrs_relocate_resource2(clscrs_splist *resid_list, 
                          const oratext *destnode,
                          clscrs_env env,
                          clscrs_msgf2 msgf, void *msgarg, uword flags, 
                          clscrs_reslist *op_status);

/*
  NAME
    clscrs_relocate_server

  DESCRIPTION
    Relocate a list of servers. 

    The destination serverpool is the serverpool to which the servers need to be
    relocated. The destination serverpool is a required parameter.

  PARAMETERS
    server_list  [IN]  -  servers to relocate
    destsrvpool  [IN]  -  destination serverpool to relocate to, cannot be null
    env          [IN]  -  environment arguments to the stop operation
    msgf         [IN]  -  user message callback, may be null.
    msgarg       [IN]  -  user callback argument, may be null.
    flags        [IN]  -  async, queue flags
    op_status    [OUT] -  server list holding the status of the relocate
                          operation for each server.  

  RETURNS
    CLSCRS_STAT_INVALID_RESNAME       - if server_list is empty
    CLSCRS_STAT_AUTHORIZATION_FAILURE - on authorization failure
    CLSCRS_STAT_SUCCESS               - if the servers were relocated 
                                        successfully
    CLSCRS_STAT_FAILURE               - if the servers were not able
                                        to be relocated
    CLSCRS_STAT_CONNECTION            - if there is a communication error
                                        with the crsd

  SINCE
    CRS - 11.2
*/

CLSCRS_STAT 
clscrs_relocate_server(clscrs_splist *server_list, 
                       const oratext *destsrvpool,
                       clscrs_env env,
                       clscrs_msgf2 msgf, void *msgarg, uword flags, 
                       clscrs_reslist *op_status);

/*
  NAME
    clscrs_compfilter_create

  DESCRIPTION
    Constructs a simple filter that compares two values. 
    Eg, attrib_name==value, attrib_name!=value.

    The memory is allocated by the function which must be freed
    by calling clscrs_filter_destroy().
    
  PARAMETERS
    ctx           [IN]   - clscrs context
    name          [IN]   - left operand of the filter
    comparator    [IN]   - the comparison operator enumeration
    value         [IN]   - right operand of the filter
    simple_filter [OUT]  - the simple comparison filter

  RETURNS
    clscrsretBADCTX  - if context is null 
    clscrsretBADNAME - if name or value is null
    clscrsretBADARG  - if name, value or comparator is invalid
    clscrsretSUCC    - on success
*/
clscrsret
clscrs_compfilter_create(clscrs_ctx *ctx,
                         const oratext *name,
                         clscrs_comparator comparator, 
                         const oratext *value, 
                         clscrs_filter **simple_filter);


/*
  NAME
    clscrs_exprfilter_create

  DESCRIPTION
    Constructs an expression filter from comparison and/or expression filters.
    Eg, (attrib_name1==value1 && attrib_name2!=value2)

    The memory is allocated by the function which must be freed
    by calling clscrs_filter_destroy().
    
  PARAMETERS
    filter1     [IN]  -  left filter operand
    operator    [IN]  -  the expression operator enumeration
    filter2     [IN]  -  right filter operand
    expr_filter [OUT] -  the expression filter

  RETURNS
    clscrsretBADARG  - if filter1 or filter2 is null
                       if operator is invalid
    clscrsretSUCC    - on success
*/
clscrsret
clscrs_exprfilter_create(clscrs_filter *filter1, 
                         clscrs_operator filter_operator, 
                         clscrs_filter *filter2, 
                         clscrs_filter **expr_filter);

/*
  NAME
    clscrs_filter_print

  DESCRIPTION
    Print the filter

  PARAMETERS
     filter [IN]  -  filter to print

  RETURNS
    clscrsretBADARG - if filter is null
    clscrsretSUCC   - on success
*/
clscrsret 
clscrs_filter_print(clscrs_filter *filter);

/*
  NAME
    clscrs_filter_destroy

  DESCRIPTION
    Free the memory for a filter.

  PARAMETERS
     filter [IN]  -  filter to destroy

  RETURNS
    clscrsretSUCC - on success
*/

clscrsret clscrs_filter_destroy(clscrs_filter **filter);

/*
  NAME
    clscrs_filter_to_string

  DESCRIPTION
    Converts a filter to its string representation where the 
    individual elements of the filter are separated by 
    CLSCRS_ATTRIBUTE_DELIMITER.
    E.g., LOCATION == HUB is converted to ^ALOCATION^A==^AHUB^A

  PARAMETERS
    filter  [IN]     - filter whose string representation is needed
    str_rep [OUT]    - string representation of the filter
    buffer  [IN/OUT] - size of the buffer for str_rep
                       If the size of the buffer given is smaller than what
                       is required, it will return the value of the buffer 
                       size needed. 
  RETURNS
   clscrsretLONGNAME   - if left operand is more than CLSCRS_ENTITYNAME_MAXLEN
   clscrsretSUCC       - on success
   clscrsretNOMEM      - if memory could not be allocated
   clscrsretBADARG     - on either of str_rep being NULL, or buffer being
                       - NULL, or filter being NULL
   clscrsretBUFFSMALL  - if the buffer size passed in, is smaller than
                         what is required
*/
clscrsret
clscrs_filter_to_string(clscrs_filter *filter, 
                        oratext       *str_rep,
                        size_t        *buffer);

/*
  NAME
    clscrs_string_to_filter

  DESCRIPTION
    Converts a string to a clscrs_filter entity.
    The user is responsible for freeing the memory that is allocated
    for the filter.
    NOTE: The comparators and operators that should be specified in the 
    string should be from the filter constants defined in this header file.
    E.g.: A string that is passed should look like:
        ((CPU_COUNT > 4) || (CPU_CLOCK_RATE == 10)), or
        ((NAME co austin) && (TYPE == ora.db.type))

  PARAMETERS
    ctx     [IN]     - clscrs context
    str     [IN]     - string for which you need the filter
    filter  [OUT]    - filter representation of the input string

  RETURNS
   clscrsretSUCC       - on success
   clscrsretBADARG     - if str is NULL, or the input string is malformed
*/
clscrsret
clscrs_string_to_filter(clscrs_ctx *ctx, oratext *str,
                        clscrs_filter **filter); 

/*
  NAME
    clscrs_is_crs_admin

  DESCRIPTION
    Checks if the user is a CRS Admin. 

    NOTE: Since 12.1, this API will need a running CRS stack to operate.

  PARAMETERS
     user_name [IN]  -  user name to be checked
     is_admin  [OUT] -  sets it to true if the user is part of CRSADMIN
                        or CRSUSER, false otherwise

  RETURNS
    clscrsretBADARG - if user_name is null
    clscrsretSUCC   - on success
    clscrsretFAIL   - on any failure
*/
clscrsret 
clscrs_is_crs_admin(const oratext* user_name, boolean* is_admin);

/*
  NAME
    clscrs_get_fixed_attrlist

  DESCRIPTION
  Get the list of attributes corresponding to an attribute group identifier.

  PARAMETERS
    ctx      [IN]    - clscrs context
    attrgrp  [IN]    - attribute group that identifies a group of attributes
    attrlist [OUT]   - list of attributes returned that corresponds to attrgrp

  RETURNS
    clscrsretSUCC - on success
*/

clscrs_splist* clscrs_get_fixed_attrlist(clscrs_ctx *ctx, 
                                         clscrs_attr_grp attrgrp);


/*
  NAME
    clscrs_get_entity_type

  DESCRIPTION
    Gets the entity type corresponding to the entity id provided

  PARAMETERS
    entity_id   [IN]    - id that needs to be identified
    entity_type [OUT]   - one of resource/type/sp/server/resinst

  RETURNS
    clscrsretSUCC   - on success
    clscrsretBADARG - on either of the input parameters being NULL
*/

clscrsret clscrs_get_entity_type(const oratext* entity_id, 
                                 clscrs_entity* entity_type);


/*
  NAME
    clscrs_get_resource_instance_details

  DESCRIPTION
    Gets the resource instance details like resource name, cardinality
    and degree from the resource instance id provided.
    NOTE: The caller is responsible for allocating memory for res_name, 
    cardinality and degree, before they are passed in.

  PARAMETERS
    resinst_id  [IN]     - id that needs to be split into its details
    res_name    [OUT]    - name of the resource
    cardinality [OUT]    - cardinality id, or server name if its a local resource
    degree      [OUT]    - degree of the resource instance
    buffer      [IN/OUT] - size of the buffer for res_name, cardinality, degree.
                           If the size of the buffer given is smaller than what
                           is required, it will return the value of the buffer 
                           size needed. 

  RETURNS
   clscrsretSUCC       - on success
   clscrsretNOMEM      - if memory could not be allocated
   clscrsretBADARG     - on either of resinst_id being NULL, or buffer being
                       - NULL, or all the three: res_name, cardinality and 
                       - degree being NULL
   clscrsretBADRESNAME - if resinst_id is not a resource instance id
   clscrsretBUFFSMALL  - if the buffer size passed in, is smaller than
                         what is required
                              
*/

clscrsret clscrs_get_resource_instance_details(const oratext* resinst_id,
                                               oratext* res_name,
                                               oratext* cardinality,
                                               oratext* degree,
                                               size_t* buffer);

/* ------------------------------- clscrs_stat3 ---------------------------- */
/*
  NAME
    clscrs_stat3

  DESCRIPTION
    Get information about the CRS entities identified in qlist. Information 
    is returned in the output out_entitylist. Queries are executed such that
    the resultant output set is consistent.

    User has to create clscrs_query objects and append it to the
    clscrs_querylist object before passing it to this function. Any attributes
    required for each entity type need to be present in the clscrs_query object.
    If the attribute list is null, all attributes will be returned.

    The out_entitylist must be created and passed as an empty list.
    Any errors for an entity are returned in the out_entitylist.

    The output is returned as a nested entitylist. Results for individual
    queries are returned as respective entities in the out_entitylist. The o/p
    for individual queries are returned as an entitylist for that specific
    entity. The type of results in the entity can be verified by calling 
    clscrs_crsentity_get_type, to get the crs entity type of the entity that is
    part of the out_entitylist.

    E.g., If you have 2 queries, one for resources and the other for resource
    types, out_entitylist will contain 2 entity objects, one for each of the 
    above crs entity types. The entitylist for each of these entity objects 
    will contain the results of the queries. To figure out what kind of entity
    object a particular one is, the user needs to call clscrs_crsentity_get_type
    function on that entity object. If the query returns zero matches, the size
    of the entitylist for that entity object will be zero.

    NOTE: Only one clscrs_query object per entity type is allowed. If more than
    one clscrs_query object is created for the same entity type, stat3
    behavior will not be guaranteed.

  PARAMETERS
    qlist           [IN]    list of CRS entities to query
    flags           [IN]    flags 
    out_entitylist  [OUT]   list holding the returned entity information

  FLAGS
    Currently, no flags are supported. 

  RETURNS
  
    CLSCRS_STAT_SUCCESS     - if the API is able to successfully query the CRSD 
    CLSCRS_STAT_FAILURE     - if there was any error in querying the CRSD
    CLSCRS_STAT_CONNECTION  - if there is an error communicating with the crsd 

  SINCE
    CRS - 12.1
*/
CLSCRS_STAT
clscrs_stat3(clscrs_querylist      *qlist, 
             uword                  flags,
             clscrs_crsentitylist  *out_entitylist);


/*
  NAME
    clscrs_query_create

  DESCRIPTION
    Create a new query structure. The memory for the query structure is allocated 
    by the function. The memory is freed when a query list (clscrs_querylist) 
    is destroyed through clscrs_querylist_destroy().

  PARAMETERS
    ctx     [IN]  - clscrs context
    type    [IN]  - entity type for which this query structure is created
    query   [OUT] - the new query structure created

  RETURNS
    clscrsretSUCC    - on success
    clscrsretNOMEM   - if no memory can be allocated
    clscrsretBADCTX  - if the context is null
*/

clscrsret clscrs_query_create(clscrs_ctx *ctx, clscrs_entity_type type,
                              clscrs_query **query);


/*
  NAME
    clscrs_query_destroy

  DESCRIPTION
    Free the memory for a query structure.

  PARAMETERS
    query      [IN] - query structure for which the memory is freed

  RETURNS
    clscrsretSUCC - on success
*/

clscrsret clscrs_query_destroy(clscrs_query **query);


/*
  NAME
    clscrs_query_set_attr

  DESCRIPTION
    Set a CRS entity attribute for a clscrs_query structure.

  PARAMETERS
    query    [IN] - query structure for which the attribute is set
    attrname [IN] - name of the CRS entity attribute
    value    [IN] - value for the CRS entity attribute (may be NULL)

  RETURNS
    clscrsretSUCC   - on success
    clscrsretBADARG - if the attribute name is null
    clscrsretNOMEM  - if no memory can be allocated
*/

clscrsret clscrs_query_set_attr(clscrs_query *query,const oratext *attrname,
                                const oratext *value);


/*  
  NAME
    clscrs_query_set_attr_list

  DESCRIPTION
    Set the attribute list for an entity in a query structure. The attribute list is a list of 
    stringpairs. The list is created from the clscrs_splist_create call.

  PARAMETERS
    query    [IN]  - query structure for which the attribute list is set
    attrlist [IN]  - list of attributes to be set for the given entity in the query structure

  RETURNS
    clscrsretSUCC   - on success  
    clscrsretBADARG - if the query structure is null
*/

clscrsret clscrs_query_set_attr_list(clscrs_query *query, clscrs_splist *attrlist);


/*  
  NAME
    clscrs_query_set_filter

  DESCRIPTION
    Set the filter for an entity in a query structure. 

  PARAMETERS
    query    [IN]  - query structure for which the filter is set
    filter   [IN]  - filter specific to the entity specified in the query structure

  RETURNS
    clscrsretSUCC   - on success  
    clscrsretBADARG - if the query structure is null
*/

clscrsret clscrs_query_set_filter(clscrs_query *query, clscrs_filter *filter);


/*
  NAME
    clscrs_querylist_create

  DESCRIPTION
    Create a new query list. The memory for the query list is allocated 
    by the function.

  PARAMETERS
    ctx       [IN]  - clscrs context
    querylist [OUT] - querylist (empty) that is created

  RETURNS
    clscrsretSUCC   - on success
    clscrsretNOMEM  - if no memory can be allocated
    clscrsretBADCTX - if the context is null
*/

clscrsret clscrs_querylist_create(clscrs_ctx *ctx, clscrs_querylist **querylist);

/*
  NAME
    clscrs_querylist_append

  DESCRIPTION
    Add a clscrs_query to a query list.

  PARAMETERS
    querylist [IN] - query list to add the query to
    query     [IN] - query to add

  RETURNS
    clscrsretSUCC      - on success
    clscrsretBADARG    - if querylist is null
    clscrsretRESEXISTS - if the query already exists in the querylist
*/

clscrsret clscrs_querylist_append(clscrs_querylist *querylist, clscrs_query *query);

/*
  NAME
    clscrs_querylist_first

  DESCRIPTION
    Get the first query from a query list. 

  PARAMETERS
    querylist [IN]     - query list for which the first query is to be obtained
    query     [OUT] - the first query on the query list

  RETURNS
    clscrsretSUCC   - on success.
    clscrsretBADARG - if querylist is null
    clscrsretEMPTY  - if there are no query structures in the list
*/

clscrsret clscrs_querylist_first(clscrs_querylist *querylist, clscrs_query **query);

/*
  NAME
    clscrs_querylist_next

  DESCRIPTION
    Get the next query from the query list. This function is 
    called to iterate over the query structures in a query list.

  PARAMETERS
    querylist [IN]  - query list for which the next query is to be obtained
    query     [OUT] - the next query on the query list

  RETURNS
    clscrsretSUCC    - on success
    clscrsretBADARG  - if query list is null
    clscrsretENDLIST - if there are no more query structures in the list
*/

clscrsret clscrs_querylist_next(clscrs_querylist *querylist, clscrs_query **query);

/*
  NAME
    clscrs_querylist_count

  DESCRIPTION
    Count the number of query structures in a query list.

  PARAMETERS
    querylist [IN]  - query list for which the count is needed
    count     [OUT] - number of query structures in the query list

  RETURNS
    clscrsretSUCC   - on success
    clscrsretBADARG - if querylist is null
*/

clscrsret clscrs_querylist_count(clscrs_querylist *querylist, ub4 *count);

/*
  NAME
    clscrs_querylist_destroy

  DESCRIPTION
    Free the memory for a query list.

  PARAMETERS
    querylist [IN]  - query list for which the memory is to be freed

  RETURNS
    clscrsretSUCC - on success
*/

clscrsret clscrs_querylist_destroy(clscrs_querylist **querylist);


/*---------- New and Wrapper Functions for clscrs_res* ----------------------*/

/* clscrs_crsentity_create */
clscrsret clscrs_crsentity_create(clscrs_ctx *ctx, const oratext *entityname, 
                                  clscrs_crsentity **entity);

/* clscrs_crsentity_get_name */
clscrsret clscrs_crsentity_get_name(clscrs_crsentity *entity, oratext **name);

/* clscrs_crsentity_set_attr */
clscrsret clscrs_crsentity_set_attr(clscrs_crsentity *entity, const oratext *attrname, 
                                    const oratext *value);

/* clscrs_crsentity_get_attr*/
clscrsret clscrs_crsentity_get_attr(clscrs_crsentity *entity, const oratext *attrname, 
                                    oratext **value);

/* clscrs_crsentity_get_attr_list*/
clscrsret clscrs_crsentity_get_attr_list(clscrs_crsentity *entity,
                                         clscrs_splist **attrlist);

/* clscrs_crsentity_set_attr_list*/
clscrsret clscrs_crsentity_set_attr_list(clscrs_crsentity *entity, 
                                         clscrs_splist *attrlist);

/* clscrs_crsentity_attr_count */
clscrsret clscrs_crsentity_attr_count(clscrs_crsentity *entity, ub4 *count);

/*
  NAME
    clscrs_crsentity_get_crsentitylist

  DESCRIPTION
    Get the entitylist for a crs entity by name.
    If name is NULL, behavior will be same as clscrs_res_get_reslist,
    i.e., it will return only the top-level entitylist associated with
    the crs entity. 
    NOTE: Passing of NULL for name is only allowed for backward compatability.

  PARAMETERS
    entity        [IN] - crs entity for which the entitylist is set
    crsentitylist [IN] - entity list to be returned for the given crs entity
    name          [IN] - get the entity list by name. This can be NULL.

  RETURNS
    clscrsretSUCC   - on success  
    clscrsretBADARG - if the entity is null
*/
clscrsret clscrs_crsentity_get_crsentitylist(clscrs_crsentity *entity, 
                                             clscrs_crsentitylist**
                                             crsentitylist,
                                             const oratext *name);

/* clscrs_crsentity_set_crsentitylist */
clscrsret clscrs_crsentity_set_crsentitylist(clscrs_crsentity *entity,
                                             clscrs_crsentitylist*
                                             crsentitylist);

/* clscrs_crsentity_destroy */
clscrsret clscrs_crsentity_destroy(clscrs_crsentity **entity);

/* clscrs_crsentity_get_node_list */
clscrsret clscrs_crsentity_get_node_list(clscrs_crsentity *entity, 
                                         clscrs_splist **nodelist);

/* clscrs_crsentity_get_op_status */
clscrsret clscrs_crsentity_get_op_status(clscrs_crsentity *entity, 
                                         CLSCRS_STAT *status, 
                                         oratext **msg);

/* clscrs_crsentity_get_registered */
clscrsret clscrs_crsentity_get_registered(clscrs_crsentity *entity, 
                                          boolean *registered);

/*
  NAME
    clscrs_crsentity_get_type

  DESCRIPTION
    Gets the entity type corresponding to the crs entity provided

  PARAMETERS
    entity      [IN]    - crs entity that needs to be identified
    entity_type [OUT]   - one of resource/type/sp/server/resinst/
                          category/policy/policyset

  RETURNS
    clscrsretSUCC   - on success
    clscrsretBADARG - on either of the input parameters being NULL
*/

clscrsret clscrs_crsentity_get_type(clscrs_crsentity   *entity, 
                                    clscrs_entity_type *entity_type);

/*
  NAME
    clscrs_crsentity_first_entitylist

  DESCRIPTION
    Get the first entitylist for the crs entity passed in.

  PARAMETERS
    in_entity      [IN]  - crs entity for which the first entitylist is needed
    out_entitylist [OUT] - first entity list for the crs entity

  RETURNS
    clscrsretSUCC   - on success.
    clscrsretBADARG - if in_entity is null
    clscrsretEMPTY  - if there are no entity lists for the crs entity
*/
clscrsret 
clscrs_crsentity_first_entitylist(clscrs_crsentity      *in_entity, 
                                  clscrs_crsentitylist **out_entitylist);

/*
  NAME
    clscrs_crsentity_next_entitylist

  DESCRIPTION
    Get the next entity list for the crs entity passed in. This function is 
    called to iterate over the entity lists for a given crs entity.

  PARAMETERS
    in_entity      [IN]  - crs entity for which the next entitylist is needed
    out_entitylist [IN/OUT] - next entity list for the crs entity

  RETURNS
    clscrsretSUCC    - on success
    clscrsretBADARG  - if in_entity or out_entitylist is null
    clscrsretENDLIST - if there are no more entity lists for the crs entity
*/
clscrsret 
clscrs_crsentity_next_entitylist(clscrs_crsentity      *in_entity,
                                 clscrs_crsentitylist **out_entitylist);

/*
  NAME
    clscrs_crsentity_append_entitylist

  DESCRIPTION
    Add an entity list to a crs entity.

  PARAMETERS
    in_entity  [IN] - crs entity to add the entitylist to
    entitylist [IN] - entity list to add

  RETURNS
    clscrsretSUCC      - on success
    clscrsretBADARG    - if in_entity or entitylist is null
    clscrsretRESEXISTS - if the entitylist already exists in the crs entity
*/
clscrsret 
clscrs_crsentity_append_entitylist(clscrs_crsentity     *in_entity,
                                   clscrs_crsentitylist *entitylist);

/*
  NAME
    clscrs_crsentity_count_entitylist

  DESCRIPTION
    Count the number of entity list in the crs entity.

  PARAMETERS
    in_entity [IN]  - crs entity for which the count is needed
    count     [OUT] - number of entity lists in the crs entity

  RETURNS
    clscrsretSUCC   - on success
    clscrsretBADARG - if in_entity is null
*/
clscrsret 
clscrs_crsentity_count_entitylist(clscrs_crsentity *in_entity, 
                                  ub4 *count);

/*
  NAME
    clscrs_crsentity_delete_entitylist

  DESCRIPTION
    Delete en entitylist from the crs entity by name.

  PARAMETERS
    in_entity [IN] - crs entity from which to delete the entitylist
    name      [IN] - name of the entitylist to delete 

  RETURNS
    clscrsretSUCC   - on success
    clscrsretBADARG - if in_entity or name is null
    clscrsretNORES  - if the entitylist is not there in the crs entity
*/
clscrsret 
clscrs_crsentity_delete_entitylist(clscrs_crsentity *in_entity,
                                   const oratext     *name);
/**** New and Wrapper functions for clscrs_reslist* functions ****/

/*
  NAME
    clscrs_crsentitylist_create

  DESCRIPTION
    Create a new crsentity list. The memory for the entity list is allocated 
    by the function.
    If name is NULL, behavior will be same as clscrs_reslist_create.
    NOTE: Passing of NULL for name is only allowed for backward compatability.
    It will be discontinued in the future. 

  PARAMETERS
    ctx           [IN]  - clscrs context
    crsentitylist [OUT] - crsentity list (empty) that is created
    name          [IN]  - name of the crsentitylist created

  RETURNS
    clscrsretSUCC   - on success
    clscrsretNOMEM  - if no memory can be allocated
    clscrsretBADCTX - if the context is null
*/
clscrsret clscrs_crsentitylist_create(clscrs_ctx *ctx, 
                                      clscrs_crsentitylist **crsentitylist,
                                      const oratext *name);

/* clscrs_crsentitylist_append */
clscrsret clscrs_crsentitylist_append(clscrs_crsentitylist *crsentitylist,
                                      clscrs_crsentity *entity);

/* clscrs_crsentitylist_first */
clscrsret clscrs_crsentitylist_first(clscrs_crsentitylist *crsentitylist, 
                                     clscrs_crsentity **entity);

/* clscrs_crsentitylist_next */
clscrsret clscrs_crsentitylist_next(clscrs_crsentitylist *crsentitylist,
                                    clscrs_crsentity **entity);

/* clscrs_crsentitylist_find */
clscrsret clscrs_crsentitylist_find(clscrs_crsentitylist *crsentitylist, 
                                    const oratext *name, 
                                    clscrs_crsentity **entity);

/* clscrs_crsentitylist_findtype */
clscrsret clscrs_crsentitylist_findtype(clscrs_crsentitylist *crsentitylist, 
                                        clscrs_entity_type type,
                                        clscrs_crsentity **entity);

/* clscrs_crsentitylist_count */
clscrsret clscrs_crsentitylist_count(clscrs_crsentitylist *crsentitylist, 
                                     ub4 *count);

/* clscrs_crsentitylist_delete_crsentity */
clscrsret 
clscrs_crsentitylist_delete_crsentity(clscrs_crsentitylist *crsentitylist,
                                      const oratext *name);

/* clscrs_crsentitylist_destroy */
clscrsret clscrs_crsentitylist_destroy(clscrs_crsentitylist **crsentitylist);

/* clscrs_crsentitylist_get_seqnumber */
clscrsret clscrs_crsentitylist_get_seqnumber(clscrs_crsentitylist *crsentitylist, 
                                   sb4 *seqnum);

/* clscrs_crsentitylist_isempty */
clscrsret clscrs_crsentitylist_isempty(clscrs_crsentitylist *crsentitylist);

/* clscrs_crsentitylist_set_seqnumber */
clscrsret clscrs_crsentitylist_set_seqnumber(clscrs_crsentitylist *crsentitylist, 
                                             sb4 seqnum);

/*
  NAME
    clscrs_crsentitylist_get_name

  DESCRIPTION
    Get the name of an entitylist

  PARAMETERS
    res   [IN]  - entitylist for which the name is needed
    name  [OUT] - name of the entitylist

  RETURNS
    clscrsretSUCC    - on success
    clscrsretBADARG  - if the in_entitylist argument is null
*/
clscrsret 
clscrs_crsentitylist_get_name(clscrs_crsentitylist *in_entitylist, 
                              oratext **name);

/*------------------------ What-If functions -------------------------------*/

/*
  NAME
    clscrs_whatif_register_resource

  DESCRIPTION
    Determines the actions that CRS would take if the resource in question
    was added or modified.

  PARAMETERS
    name   [IN]  -  Name of the resource
    attrs  [IN]  -  The attributes for the specified resource.
    flags  [IN]  -  CLSCRS_FLAG_NONE,
                    CLSCRS_FLAG_WHATIF_VERBOSE,
                    CLSCRS_FLAG_FORCE
		    CLSCRS_FLAG_REG_UPDATE (to modify the resource)
                    See CLSCRS_FLAG enum for more details.    
    qlist  [IN]  -  specifies the client query on CRS entity status. 
                    NULL indicates no query specified.

    status [OUT] -  list holding the returned entity information. 
    alist  [OUT] -  action list holding the actions that would be performed. 

  RETURNS
    CLSCRS_STAT_SUCCESS               - if the request completed successfully
    CLSCRS_STAT_AUTHORIZATION_FAILURE - on authorization failure
    CLSCRS_STAT_CONNECTION            - if there is a communication error
    CLSCRS_STAT_WRONG_ACTIVE_VERSION  - if used before the cluster has been                     
                                        upgraded to 12
    CLSCRS_STAT_INTERNAL_ERROR        - if unexpected non-user error 
    CLSCRS_STAT_INVALID_ARGS          - if the arguments provided are incorrect

  SINCE
    CRS - 12.1
*/

CLSCRS_STAT 
clscrs_whatif_register_resource(const oratext *name, clscrs_splist *attrs, 
				uword flags, clscrs_querylist *qlist,
				clscrs_crsentitylist *status,
				clscrs_actionlist *alist);

/*
  NAME
    clscrs_whatif_fail_resource

  DESCRIPTION
    Determines the actions that CRS would take if the resource(s) in question 
    fail.

  PARAMETERS
    name     [IN]  -  Resource or instance id or a filter
    server   [IN]  -  Server name on which the resource failure occurs. 
                      NULL is allowed.
    flags    [IN]  -  CLSCRS_FLAG_NONE
                      CLSCRS_FLAG_WHATIF_VERBOSE,
                      See CLSCRS_FLAG enum for more details.
    qlist    [IN]  -  Specifies the client query on CRS entity status
                      NULL indicates no query specified.
    alist    [OUT] -  action list holding the actions that would be performed. 
    status   [OUT] -  list holding the returned entity information. 

  RETURNS
    CLSCRS_STAT_SUCCESS               - if the request completed successfully
                                        or if no entities matched a filter
    CLSCRS_STAT_AUTHORIZATION_FAILURE - on authorization failure
    CLSCRS_STAT_CONNECTION            - if there is a communication error
    CLSCRS_STAT_WRONG_ACTIVE_VERSION  - if used before the cluster has been                     
                                        upgraded to 12
    CLSCRS_STAT_INTERNAL_ERROR        - if unexpected non-user error 
    CLSCRS_STAT_INVALID_ARGS          - if the arguments provided are incorrect

  SINCE
    CRS - 12.1
*/
CLSCRS_STAT 
clscrs_whatif_fail_resource(clscrs_splist *name, const oratext *server,
                            uword flags, clscrs_querylist *qlist, 
                            clscrs_crsentitylist *status, 
                            clscrs_actionlist *alist);

/*
  NAME
    clscrs_whatif_start_resource

  DESCRIPTION
    Determines the actions that CRS would take if the resource(s) in question
    were started.

  PARAMETERS
    name    [IN]  -  Resource or instance id or a filter
    node    [IN]  -  the nodename on which to start the resource, may be null
    flags   [IN]  -  CLSCRS_FLAG_NONE,
                     CLSCRS_FLAG_WHATIF_VERBOSE,
                     CLSCRS_FLAG_FORCE
                     See CLSCRS_FLAG enum for more details.
    qlist   [IN]  -  specifies the client query on CRS entity status. 
                     NULL indicates no query specified.
    status  [OUT] -  list holding the returned entity information. 
    alist   [OUT] -  action list holding the actions that would be performed. 

  RETURNS
    CLSCRS_STAT_SUCCESS               - if the request completed successfully
                                        or if no entities matched a filter
    CLSCRS_STAT_AUTHORIZATION_FAILURE - on authorization failure
    CLSCRS_STAT_CONNECTION            - if there is a communication error
    CLSCRS_STAT_WRONG_ACTIVE_VERSION  - if used before the cluster has been                     
                                        upgraded to 12
    CLSCRS_STAT_INTERNAL_ERROR        - if unexpected non-user error 
    CLSCRS_STAT_INVALID_ARGS          - if the arguments provided are incorrect

  SINCE
    CRS - 12.1
*/
CLSCRS_STAT 
clscrs_whatif_start_resource(clscrs_splist *name, const oratext *node,
                             uword flags, clscrs_querylist *qlist,
                             clscrs_crsentitylist *status,
                             clscrs_actionlist *alist);


/*
  NAME
    clscrs_whatif_stop_resource

  DESCRIPTION
    Determines the actions that CRS would take if the resource(s) in question
    were stopped.

  PARAMETERS
    name    [IN]  -  resource or instance id or a filter
    flags   [IN]  -  CLSCRS_FLAG_NONE,
                     CLSCRS_FLAG_WHATIF_VERBOSE,
                     CLSCRS_FLAG_FORCE
                     See CLSCRS_FLAG enum for more details.
    qlist   [IN]  -  specifies the client query on CRS entity status
                     NULL indicates no query specified.
    status  [OUT] -  list holding the returned entity information. 
    alist   [OUT] -  action list holding the actions that would be performed. 

  RETURNS
    CLSCRS_STAT_SUCCESS               - if the request completed successfully
                                        or if no entities matched a filter
    CLSCRS_STAT_AUTHORIZATION_FAILURE - on authorization failure
    CLSCRS_STAT_CONNECTION            - if there is a communication error
    CLSCRS_STAT_WRONG_ACTIVE_VERSION  - if used before the cluster has been                     
                                        upgraded to 12
    CLSCRS_STAT_INTERNAL_ERROR        - if unexpected non-user error 
    CLSCRS_STAT_INVALID_ARGS          - if the arguments provided are incorrect

  SINCE
    CRS - 12.1
*/
CLSCRS_STAT 
clscrs_whatif_stop_resource(clscrs_splist *name, uword flags, 
                            clscrs_querylist *qlist, 
                            clscrs_crsentitylist *status,
                            clscrs_actionlist *alist);

/*
  NAME
    clscrs_whatif_relocate_resource

  DESCRIPTION
    Determines the actions that CRS would take if the resource(s) in question
    were relocated.

  PARAMETERS
    name     [IN]  -  resource or instance id or a filter
    destnode [IN]  -  the nodename on which to start the resource
    flags    [IN]  -  CLSCRS_FLAG_NONE,
                      CLSCRS_FLAG_WHATIF_VERBOSE,
                      CLSCRS_FLAG_FORCE
                      See CLSCRS_FLAG enum for more details.
    qlist    [IN]  -  specifies the client query on CRS entity status
                      NULL indicates no query specified.
    alist    [OUT] -  action list holding the actions that would be performed. 
    status   [OUT] -  list holding the returned entity information. 

  RETURNS
    CLSCRS_STAT_SUCCESS               - if the request completed successfully
                                        or if no entities matched a filter
    CLSCRS_STAT_AUTHORIZATION_FAILURE - on authorization failure
    CLSCRS_STAT_CONNECTION            - if there is a communication error
    CLSCRS_STAT_WRONG_ACTIVE_VERSION  - if used before the cluster has been                     
                                        upgraded to 12
    CLSCRS_STAT_INTERNAL_ERROR        - if unexpected non-user error 
    CLSCRS_STAT_INVALID_ARGS          - if the arguments provided are incorrect

  SINCE
    CRS - 12.1
*/
CLSCRS_STAT 
clscrs_whatif_relocate_resource(clscrs_splist *name, const oratext *destnode,
                                uword flags, clscrs_querylist *qlist, 
                                clscrs_crsentitylist *status, 
                                clscrs_actionlist *alist);

/*
  NAME
    clscrs_whatif_register_serverpool

  DESCRIPTION
    Determines the actions that CRS would take if the serverpool as specified
    were registered.

  PARAMETERS
    pname    [IN]  -  name of the server pool
    attrs    [IN]  -  attributes for the pool to be registered
    flags    [IN]  -  CLSCRS_FLAG_NONE,
                      CLSCRS_FLAG_WHATIF_VERBOSE,
                      CLSCRS_FLAG_FORCE
		      CLSCRS_FLAG_REG_UPDATE (to modify the serverpool)
                      See CLSCRS_FLAG enum for more details.
    qlist    [IN]  -  specifies the client query on CRS entity status
                      NULL indicates no query specified.
    alist    [OUT] -  action list holding the actions that would be performed. 
    status   [OUT] -  list holding the returned entity information. 

  RETURNS
    CLSCRS_STAT_SUCCESS               - if the request completed successfully
                                        or if no entities matched a filter
    CLSCRS_STAT_AUTHORIZATION_FAILURE - on authorization failure
    CLSCRS_STAT_CONNECTION            - if there is a communication error
    CLSCRS_STAT_WRONG_ACTIVE_VERSION  - if used before the cluster has been                     
                                        upgraded to 12
    CLSCRS_STAT_INTERNAL_ERROR        - if unexpected non-user error 
    CLSCRS_STAT_INVALID_ARGS          - if the arguments provided are incorrect

  SINCE
    CRS - 12.1
*/
CLSCRS_STAT 
clscrs_whatif_register_serverpool(const oratext *pname, clscrs_splist *attrs, 
                                  uword flags, clscrs_querylist *qlist, 
                                  clscrs_crsentitylist *status, clscrs_actionlist *alist);

/*
  NAME
    clscrs_whatif_unregister_serverpool

  DESCRIPTION
    Determines the actions that CRS would take if the specified serverpool
    were unregistered.

  PARAMETERS
    pname    [IN]  -  name of the server pool
    flags    [IN]  -  CLSCRS_FLAG_NONE,
                      CLSCRS_FLAG_WHATIF_VERBOSE,
                      CLSCRS_FLAG_FORCE
                      See CLSCRS_FLAG enum for more details.
    qlist    [IN]  -  specifies the client query on CRS entity status
                      NULL indicates no query specified.
    alist    [OUT] -  action list holding the actions that would be performed. 
    status   [OUT] -  list holding the returned entity information. 

  RETURNS
    CLSCRS_STAT_SUCCESS               - if the request completed successfully
                                        or if no entities matched a filter
    CLSCRS_STAT_AUTHORIZATION_FAILURE - on authorization failure
    CLSCRS_STAT_CONNECTION            - if there is a communication error
    CLSCRS_STAT_WRONG_ACTIVE_VERSION  - if used before the cluster has been                     
                                        upgraded to 12
    CLSCRS_STAT_INTERNAL_ERROR        - if unexpected non-user error 
    CLSCRS_STAT_INVALID_ARGS          - if the arguments provided are incorrect

  SINCE
    CRS - 12.1
*/
CLSCRS_STAT 
clscrs_whatif_unregister_serverpool(const oratext *poolname, uword flags, 
                                    clscrs_querylist *qlist, clscrs_crsentitylist *status, 
                                    clscrs_actionlist *alist);


/*
  NAME
    clscrs_whatif_add_server

  DESCRIPTION
    Determines the actions that CRS would take if the server as specified
    was added.

  PARAMETERS
    name     [IN]  -  Name of the server
    attrs    [IN]  -  The list of attributes for the server
    flags    [IN]  -  CLSCRS_FLAG_NONE,
                      CLSCRS_FLAG_WHATIF_VERBOSE,
                      CLSCRS_FLAG_FORCE
                      See CLSCRS_FLAG enum for more details.
    qlist    [IN]  -  specifies the client query on CRS entity status
                      NULL indicates no query specified.
    alist    [OUT] -  action list holding the actions that would be performed. 
    status   [OUT] -  list holding the returned entity information. 

  RETURNS
    CLSCRS_STAT_SUCCESS               - if the request completed successfully
                                        or if no entities matched a filter
    CLSCRS_STAT_AUTHORIZATION_FAILURE - on authorization failure
    CLSCRS_STAT_CONNECTION            - if there is a communication error
    CLSCRS_STAT_WRONG_ACTIVE_VERSION  - if used before the cluster has been                     
                                        upgraded to 12
    CLSCRS_STAT_INTERNAL_ERROR        - if unexpected non-user error 
    CLSCRS_STAT_INVALID_ARGS          - if the arguments provided are incorrect

  SINCE
    CRS - 12.1
*/
CLSCRS_STAT 
clscrs_whatif_add_server(const oratext *name, clscrs_splist *attrs, uword flags, 
                         clscrs_querylist *qlist, clscrs_crsentitylist *status, 
                         clscrs_actionlist *alist);

/*
  NAME
    clscrs_whatif_relocate_server

  DESCRIPTION
    Determines the actions that CRS would take if the specified server
    was moved to a new server pool.

  PARAMETERS
    name     [IN]  -  Name of the server
    topool   [IN]  -  Name of the final server pool
    flags    [IN]  -  CLSCRS_FLAG_NONE,
                      CLSCRS_FLAG_WHATIF_VERBOSE,
                      CLSCRS_FLAG_FORCE
                      See CLSCRS_FLAG enum for more details.
    qlist    [IN]  -  specifies the client query on CRS entity status
                      NULL indicates no query specified.
    alist    [OUT] -  action list holding the actions that would be performed. 
    status   [OUT] -  list holding the returned entity information. 

  RETURNS
    CLSCRS_STAT_SUCCESS               - if the request completed successfully
                                        or if no entities matched a filter
    CLSCRS_STAT_AUTHORIZATION_FAILURE - on authorization failure
    CLSCRS_STAT_CONNECTION            - if there is a communication error
    CLSCRS_STAT_WRONG_ACTIVE_VERSION  - if used before the cluster has been                     
                                        upgraded to 12
    CLSCRS_STAT_INTERNAL_ERROR        - if unexpected non-user error 
    CLSCRS_STAT_INVALID_ARGS          - if the arguments provided are incorrect

  SINCE
    CRS - 12.1
*/
CLSCRS_STAT 
clscrs_whatif_relocate_server(const oratext *name, const oratext *topool, uword flags, 
                              clscrs_querylist *qlist, clscrs_crsentitylist *status, 
                              clscrs_actionlist *alist);

/*
  NAME
    clscrs_whatif_delete_server

  DESCRIPTION
    Determines the actions that CRS would take if the specified server
    fails, or is stopped.

  PARAMETERS
    name     [IN]  -  Name of the server
    flags    [IN]  -  CLSCRS_FLAG_NONE,
                      CLSCRS_FLAG_WHATIF_VERBOSE,
                      CLSCRS_FLAG_FORCE
                      See CLSCRS_FLAG enum for more details.
    qlist    [IN]  -  specifies the client query on CRS entity status
                      NULL indicates no query specified.
    alist    [OUT] -  action list holding the actions that would be performed. 
    status   [OUT] -  list holding the returned entity information. 

  RETURNS
    CLSCRS_STAT_SUCCESS               - if the request completed successfully
                                        or if no entities matched a filter
    CLSCRS_STAT_AUTHORIZATION_FAILURE - on authorization failure
    CLSCRS_STAT_CONNECTION            - if there is a communication error
    CLSCRS_STAT_WRONG_ACTIVE_VERSION  - if used before the cluster has been                     
                                        upgraded to 12
    CLSCRS_STAT_INTERNAL_ERROR        - if unexpected non-user error 
    CLSCRS_STAT_INVALID_ARGS          - if the arguments provided are incorrect

  SINCE
    CRS - 12.1
*/
CLSCRS_STAT 
clscrs_whatif_delete_server(const oratext *name, uword flags, 
                            clscrs_querylist *qlist, clscrs_crsentitylist *status, 
                            clscrs_actionlist *alist);

/*
  NAME
    clscrs_whatif_set_activepolicy

  DESCRIPTION
    Determines the actions that CRS would take if the specified policy
    was activated.

  PARAMETERS
    name     [IN]  -  Name of the policy
    flags    [IN]  -  CLSCRS_FLAG_NONE,
                      CLSCRS_FLAG_WHATIF_VERBOSE,
                      CLSCRS_FLAG_FORCE
                      See CLSCRS_FLAG enum for more details.
    qlist    [IN]  -  specifies the client query on CRS entity status
                      NULL indicates no query specified.
    alist    [OUT] -  action list holding the actions that would be performed. 
    status   [OUT] -  list holding the returned entity information. 

  RETURNS
    CLSCRS_STAT_SUCCESS               - if the request completed successfully
                                        or if no entities matched a filter
    CLSCRS_STAT_AUTHORIZATION_FAILURE - on authorization failure
    CLSCRS_STAT_CONNECTION            - if there is a communication error
    CLSCRS_STAT_WRONG_ACTIVE_VERSION  - if used before the cluster has been                     
                                        upgraded to 12
    CLSCRS_STAT_INTERNAL_ERROR        - if unexpected non-user error 
    CLSCRS_STAT_INVALID_ARGS          - if the arguments provided are incorrect

  SINCE
    CRS - 12.1
*/
CLSCRS_STAT 
clscrs_whatif_set_activepolicy(const oratext *name, uword flags, 
                               clscrs_querylist *qlist, clscrs_crsentitylist *status, 
                               clscrs_actionlist *alist);


/* ------------- Server categorization APIs--------- */

/*
  NAME
    clscrs_servercategory_create
  
  DESCRIPTION
    Creates a new server category. The memory for the server category structure is 
    allocated by the function. The allocated memory is freed via  
    clscrs_crsentity_destroy or clscrs_crsentitylist_destroy().
  
  PARAMETERS
    ctx      [IN]  - clscrs context
    sc_name  [IN]  - name of the server category
    sc       [OUT] - the new server category created
  
  RETURNS
    clscrsretSUCC   - on success
    clscrsretNOMEM  - if no memory can be allocated
    clscrsretBADCTX - if the context is null
    clscrsretBADARG - if the resource name is null
*/
clscrsret clscrs_servercategory_create(clscrs_ctx *ctx, const oratext *sc_name,
                                 clscrs_crsentity **sc);

/*
  NAME
    clscrs_servercategory_destroy
    
  DESCRIPTION
    Free the memory for a server category structure.

  PARAMETERS
    sc      [IN] - server category for which the memory is freed

  RETURNS
    clscrsretSUCC - on success
*/
clscrsret clscrs_servercategory_destroy(clscrs_crsentity **sc);


/*
  NAME
    clscrs_register_servercategory
 
  DESCRIPTION
    Register the server categories specified the input server category list. The 
    attributes for the server category are encapsulated in the input server 
    category list. The output op_status list contains the results of the register 
    operation for each server category, and contains no valid attributes. 
    The caller must create and populate the in_entitylist and must create the 
    op_status list. Both the lists must be destroyed by the caller. The 
    op_status list cannot be reused with another api call. It must be created 
    and destroyed for each api call.
    One or more attributes of an already registered server category can be 
    modified by passing the CLSCRS_FLAG_REG_UPDATE flag. The flags apply to all 
    server categories in the input entity list.
 
 PARAMETERS
    in_entitylist  [IN]  -  list of server categories to be registered
    flags          [IN]  -  CLSCRS_FLAG_REG_UPDATE,
                            CLSCRS_FLAG_QUEUE,
                            CLSCRS_FLAG_FORCE,
                            or CLSCRS_FLAG_NONE
                            See CLSCRS_FLAG enum for more details.
    msgf           [IN]  -  user message callback, may be null.
    msgarg         [IN]  -  user callback argument, may be null.
    op_status      [OUT] -  entity list holding the status of the register
                            operation for each server category.

 RETURNS
    CLSCRS_STAT_SUCCESS      - if all input server categories are successfully 
                               registered
    CLSCRS_STAT_FAILURE      - if atleast one server category could not be 
                               registered
    CLSCRS_STAT_CONNECTION   - if there is a communication error with the crsd 
    CLSCRS_STAT_WRONG_ACTIVE_VERSION  - if used before the cluster has been upgraded to 12
    CLSCRS_STAT_INVALID_ARGS - if any of the server categories in the input entity 
                list does not have attributes 

  SINCE
    CRS - 12.1
*/

CLSCRS_STAT
clscrs_register_servercategory(clscrs_crsentitylist *in_entitylist, uword flags,
                               clscrs_msgf2 msgf, void *msgarg,
                               clscrs_crsentitylist *op_status);

/*
  NAME
    clscrs_unregister_servercategory

  DESCRIPTION
    Unregister the server categories specified in the input list. The output 
    op_status list contains the results of the unregister operation for each 
    server category.
    The caller must create and populate the sclist and must create the 
    op_status list. Both the lists must be destroyed by the caller. The 
    op_status list cannot be reused with another api call. It must be created 
    and destroyed for each api call.

  PARAMETERS
    sclist     [IN]  -  list of server category names to be unregistered
    flags      [IN]  -  option flags
    msgf       [IN]  -  user message callback, may be null.
    msgarg     [IN]  -  user callback argument, may be null.
    op_status  [OUT] -  entity list holding the status of the unregister
                        operation for each server category.

  RETURNS
    CLSCRS_STAT_SUCCESS    - if all input config policies are successfully     
                             unregistered
    CLSCRS_STAT_FAILURE    - if atleast one config policy could not be   
                             unregistered
    CLSCRS_STAT_CONNECTION - if there is a communication error with the crsd 
    CLSCRS_STAT_WRONG_ACTIVE_VERSION  - if used before the cluster has been upgraded to 12

  SINCE
    CRS - 12.1
*/
CLSCRS_STAT
clscrs_unregister_servercategory(clscrs_splist *sclist, uword flags,
                                 clscrs_msgf2 msgf, void *msgarg,
                                 clscrs_crsentitylist *op_status);


/*
  NAME
    clscrs_register_server

  DESCRIPTION
    Modify the server attributes. The attributes for the server are encapsulated in the
    input server list. The output op_status list contains the results of the modify
    operation for each server, and contains no valid attributes. 
    The caller must create and populate the in_entitylist and must create the 
    op_status list. Both the lists must be destroyed by the caller. The 
    op_status list cannot be reused with another api call. It must be created 
    and destroyed for each api call.
    NOTE: Since currently only update is supported, CLSCRS_FLAG_REG_UPDATE flag will 
    always need to be passed.
    The flags apply to all server in the input entity list.

 PARAMETERS
    in_entitylist  [IN]  -  list of server to be registered
    flags          [IN]  -  CLSCRS_FLAG_QUEUE,
                            CLSCRS_FLAG_FORCE,
                            CLSCRS_FLAG_REG_UPDATE,
                            or CLSCRS_FLAG_NONE
                            See CLSCRS_FLAG enum for more details.
    msgf           [IN]  -  user message callback, may be null.
    msgarg         [IN]  -  user callback argument, may be null.
    op_status      [OUT] -  entity list holding the status of the register
                            operation for each server.

 RETURNS
    CLSCRS_STAT_SUCCESS      - if all input server are successfully 
                               modified
    CLSCRS_STAT_FAILURE      - if atleast one server could not be     
                               modified
    CLSCRS_STAT_CONNECTION   - if there is a communication error with the crsd 
    CLSCRS_STAT_INVALID_ARGS - if any of the servers  in the input entity 
             list does not have attributes 
    CLSCRS_STAT_WRONG_ACTIVE_VERSION  - if used before the cluster has been upgraded to 12

  SINCE
    CRS - 12.1
*/    
CLSCRS_STAT
clscrs_register_server(clscrs_crsentitylist *in_entitylist, uword flags,
                       clscrs_msgf2 msgf, void *msgarg,
                       clscrs_crsentitylist *op_status);

/*
  NAME
    clscrs_get_error_details

  DESCRIPTION
    This API retrieves and returns the clsk exception stack if there are any 
    failures while invoking other CLSCRS APIs.
    NOTE: The caller is responsible for allocating memory for error_buf. 

 PARAMETERS
    error_buf  [IN]     - buffer that will be populated with the error stack 
                          details
    buf_size   [IN/OUT] - size of the buffer for error_buf.
                          If the size of the buffer given is smaller than what
                          is required, it will return the value of the buffer 
                          size needed. 

 RETURNS
   clscrsretSUCC       - if the error stack was printed successfully
   clscrsretEMPTY      - if the error stack is empty
   clscrsretBADARG     - on either of error_buf or buf_size being NULL
   clscrsretBUFFSMALL  - if the buffer size passed in, is smaller than
                         what is required

  SINCE
    CRS - 12.1
*/    
clscrsret
clscrs_get_error_details(oratext* error_buf, size_t* buf_size);

/*---------------------------------------------------------------------------
                          INTERNAL FUNCTIONS
  ---------------------------------------------------------------------------*/

#endif                                              /* CLSCRSX_ORACLE */

#ifdef __cplusplus
}
#endif /* __cplusplus */
